/**
 * This file holds the AutoDescription plugin's JS code.
 *
 * @author Sybre Waaijer https://cyberwire.nl/
 * @pluginURI https://wordpress.org/plugins/autodescription/
 *
 * @credits StudioPress for some code.
 */

// ==ClosureCompiler==
// @compilation_level SIMPLE_OPTIMIZATIONS
// @output_file_name autodescription.min.js
// @externs_url https://ajax.googleapis.com/ajax/libs/jquery/1.8.3/jquery.js
// ==/ClosureCompiler==
// http://closure-compiler.appspot.com/home

/* global autodescription, autodescriptionL10n, confirm, saveAlert, confirmReset, siteTitle, titleSep, titleSeperator, titleAdditions, titleTagline */

/* Advanced Optimizations don't work well with function tabToggle */

/**
 * Holds AutoDescription values in an object to avoid polluting global namespace.
 *
 * @since 2.2.4
 *
 * @constructor
 */
window[ 'autodescription' ] = {

	settingsChanged: false,

	titleSeperator: autodescriptionL10n.titleSep,

	titleTagline: true,

	/**
	 * Helper function for confirming a user action.
	 *
	 * @since 2.2.4
	 *
	 * @function
	 *
	 * @param {String} text The text to display.
	 * @returns {Boolean}
	 */
	confirm: function( text ) {
		'use strict';

		return confirm( text );
	},

	/**
	 * Description and title length counter, with special characters
	 *
	 * @since 2.2.4
	 *
	 * @function
	 *
	 * @return string The counter information
	 */
	updateCharacterCount: function( event ) {
		'use strict';

		jQuery( '#' + autodescription.escapeStr( event.target.id ) + '_chars' ).html( jQuery( event.target ).val().length.toString() );
	},

	/**
	 * Description and title length counter, with special characters
	 *
	 * @since 2.2.4
	 *
	 * @function
	 *
	 * @return string The counter information
	 */
	updateCharacterCountTitle: function( event ) {
		'use strict';

		var $this = jQuery( event.target ),
			$sep = autodescription.titleSeperator,
			$additions = autodescriptionL10n.titleAdditions.length,
			$siteTitle = autodescriptionL10n.siteTitle.length,
			$length = $this.val().length,
			$seplen = 3;

		// Calculate length when using HTML entities as seperator
		if ( $sep != 'pipe' && $sep != 'dash' ) {
			$seplen = $sep.length + 4;
		}

		// Tagline removed, remove additions and seperator.
		if ( ! autodescription.titleTagline ) {
			$additions = 0;
			$seplen = 0;
		}

		// Emptied input, get Site title.
		if ( $length === 0 ) {
			if ( $siteTitle === 0 ) {
				// Fetch Title from Term Title
				$length = jQuery('#name').val().length;
			} else {
				$length = $siteTitle;
			}
		}

		// Put it all together
		var $output = ( $length + $seplen + $additions ).toString();

		jQuery('#' + autodescription.escapeStr( event.target.id ) + '_chars').html( $output );
	},

	/**
	 * Set a flag, to indicate seperator field has changed.
	 *
	 * @since 2.2.4
	 *
	 * @function
	 */
	registerSepChange: function( val ) {
		'use strict';

		autodescription.titleSeperator = val;

		// @TODO create global var.
		jQuery( '#autodescription_title, #autodescription-meta-doctitle, #autodescription-meta\\[doctitle\\], #autodescription-site-settings\\[homepage_title\\]' ).trigger( 'keyup', autodescription.updateCharacterCountTitle );
	},

	/**
	 * Set a flag, to indicate seperator field has changed.
	 *
	 * @since 2.2.4
	 *
	 * @function
	 */
	registerTagChange: function( val ) {
		'use strict';

		autodescription.titleTagline = val;

		// @TODO create global var.
		jQuery( '#autodescription_title, #autodescription-meta-doctitle, #autodescription-meta\\[doctitle\\], #autodescription-site-settings\\[homepage_title\\]' ).trigger( 'keyup', autodescription.updateCharacterCountTitle );
	},

	/**
	 * Escapes HTML strings
	 *
	 * @since 2.2.4
	 *
	 * @function
	 *
	 * @return {String} HTML to jQuery converted string
	 */
	escapeStr: function( str ) {
		'use strict';

		if ( str )
		//	return str.replace(/([ #;?%&,.+*~\':"!^$[\]()=>|\/@])/g,'\\$1');
			return str.replace(/([\[\]\/])/g,'\\$1');

		return str;
	},

	/**
	 * Dynamic Title seperator replacement in metabox
	 *
	 * @since 2.2.2
	 *
	 * @function
	 *
	 * @return {Boolean} seperator has Changed
	 */
	seperatorSwitch: function( event ) {
		'use strict';

		var $sep = jQuery( ".autodescription-sep-js" ),
			$val = jQuery( event.target ).val();

		if ( $val == 'pipe' ) {
			$sep.text(" | ");
		} else if ( $val == 'dash' ) {
			$sep.text(" - ");
		} else {
			$sep.html(" &" + $val + "; ");
		}

		// Notify seperator has changed.
		return autodescription.registerSepChange( $val );
	},

	/**
	 * Status bar description output on hover
	 *
	 * @since 2.1.9
	 *
	 * @function
	 *
	 * @return {String} The information balloon
	 */
	statusBarHover: function() {
		'use strict';

		var $wrap = jQuery( '.ad-bar-wrap' ),
			$wrapA = jQuery( '.ad-bar-wrap a' );

		$wrapA.mouseenter( function() {
			var $this = jQuery( this ),
				$thisDesc = $this.attr( 'data-desc' );

			if ( ( $thisDesc !== undefined ) && ( 0 === $this.find( 'div' ).length ) ) {
				$this.append( '<div class="explanation-desc">' + $thisDesc + '<div></div></div>' );

				var $thisHeight = $this.find( 'div.explanation-desc' ).height() + 36;

				$this.find( 'div.explanation-desc' ).css( 'top', ( $this.position().top - $thisHeight ) + 'px' );
			}
		}).mousemove( function( event ) {
			var mousex = event.pageX - $wrap.offset().left;
			jQuery( 'span.ad-seo .explanation-desc div' ).css({
				left: mousex - 10 // -10px is needed, but why?
			});
		}).mouseleave( function() {
			jQuery( this ).find( 'div.explanation-desc' ).remove();
		});

	},

	/**
	 * Remove Status bar desc if clicked outside (touch support)
	 *
	 * @since 2.1.9
	 *
	 * @function
	 */
	removeDesc: function( event ) {
		'use strict';

		var $this = jQuery( event.target ),
			$desc = jQuery('.ad-bar-wrap a');

		if ( ! $this.closest( $desc ).length )
			$desc.find( 'div.explanation-desc' ).remove();
	},

	/**
	 * Refines Styling for the navigation tabs on the settings pages
	 *
	 * @since 2.2.2
	 *
	 * Rewritten
	 * @since 2.2.4
	 *
	 * @function
	 */
	tabToggle: function( event ) {
		'use strict';

		jQuery( 'div.autodescription-metaboxes .nav-tab-wrapper :input' ).each( function() {
			var $this = jQuery( this ),
				$target	= jQuery( event.target ).attr('id');

			// The second 'this' should be invalid? But it works D:
			$this.next().toggleClass( 'nav-tab-active', this.checked );

			if ( typeof $target !== 'undefined' && $target.indexOf( "-tabs-js" ) === -1 ) {
				var $id = $target;

				// Toggle content for the tabs in SEO settings page with the desired ID.
				autodescription.tabContent( $id );
			}

		});
	},

	/**
	 * Sets page output based on clicked input
	 *
	 * @since 2.2.2
	 *
	 * Rewritten
	 * @since 2.2.4
	 *
	 * @function
	 */
	tabContent: function( id ) {
		'use strict';

		if ( typeof id != 'undefined' ) {
			var $tab_id = id,
				$slice = $tab_id.slice( 0, $tab_id.indexOf('-tab') ),
				$hide = jQuery( '.' + $slice + '-tab-content' ),
				$show = jQuery( "#" + $tab_id + '-box');

			$hide.css( 'display', 'none' );
			$show.css( 'display', 'block' );
		}
	},

	/**
	 * Toggle tagline within the Left/Right example for the HomePage Title
	 *
	 * @since 2.2.4
	 *
	 * @function
	 *
	 * @return {Boolean} Title tagline removal
	 */
	taglineToggle: function( event ) {
		'use strict';

		var $this = jQuery( event.target ),
			$tag = jQuery( '.custom-blogname-js' );

		if ( $this.is(':checked') ) {
			$tag.css( 'display', 'inline' );
			return autodescription.registerTagChange( true );
		} else {
			$tag.css( 'display', 'none' );
			return autodescription.registerTagChange( false );
		}
	},

	/**
	 * Toggle tagline within the Left/Right example for the HomePage Title
	 *
	 * @since 2.2.7
	 *
	 * @function
	 *
	 * @return {Boolean} Title tagline removal
	 */
	taglineToggleOnload: function( event ) {
		'use strict';

		var $tagbox = jQuery( '#title-tagline-toggle :input' ),
			$tag = jQuery( '.custom-blogname-js' );

		if ( $tagbox.is(':checked') ) {
			$tag.css( 'display', 'inline' );
		} else {
			$tag.css( 'display', 'none' );
		}
	},

	/**
	 * Change Title based on input of the Custom Title
	 *
	 * @since 2.2.4
	 *
	 * @function
	 */
	titleProp: function( event ) {
		'use strict';

		var $val = jQuery( event.target ).val(),
			$title = jQuery( '.custom-title-js' );

		if ( $val.length === 0 ) {
			$title.text( autodescriptionL10n.siteTitle );
		} else {
			$title.text( $val );
		}
	},

	/**
	 * Have all form fields in Genesis metaboxes set a dirty flag when changed.
	 *
	 * @since 2.0.0
	 *
	 * @function
	 */
	attachUnsavedChangesListener: function() {
		'use strict';

		jQuery( 'div.autodescription-metaboxes :input' ).not('.nav-tab-wrapper :input').change( function() {
			autodescription.registerChange();
		});
		window.onbeforeunload = function(){
			if ( autodescription.settingsChanged ) {
				return autodescriptionL10n.saveAlert;
			}
		};
		jQuery( 'div.autodescription-metaboxes input[type="submit"]' ).click( function() {
			window.onbeforeunload = null;
		});
	},

	/**
	 * Set a flag, to indicate form fields have changed.
	 *
	 * @since 2.2.4
	 *
	 * @function
	 */
	registerChange: function() {
		'use strict';

		autodescription.settingsChanged = true;
	},

	/**
	 * Ask user to confirm that settings should now be reset.
	 *
	 * @since 2.2.4
	 *
	 * @function
	 *
	 * @return {Boolean} True if reset should occur, false if not.
	 */
	confirmReset: function() {
		'use strict';

		return confirm( autodescriptionL10n.confirmReset );
	},

	/**
	 * Initialises all aspects of the scripts.
	 *
	 * Generally ordered with stuff that inserts new elements into the DOM first,
	 * then stuff that triggers an event on existing DOM elements when ready,
	 * followed by stuff that triggers an event only on user interaction. This
	 * keeps any screen jumping from occuring later on.
	 *
	 * @since 2.2.4
	 *
	 * @function
	 */
	ready: function() {
		'use strict';

		// Move the page updates notices below the top-buttons
		jQuery( 'div.updated, div.error, div.notice-warning' ).insertAfter( 'p.top-buttons' );

		// Bind character counters
		jQuery( '#autodescription_description, #autodescription-meta\\[description\\], #autodescription-site-settings\\[homepage_description\\]' ).on( 'keyup', autodescription.updateCharacterCount );

		// @TODO create global var.
		jQuery( '#autodescription_title, #autodescription-meta-doctitle, #autodescription-meta\\[doctitle\\], #autodescription-site-settings\\[homepage_title\\]' ).on( 'keyup', autodescription.updateCharacterCountTitle );

		// Allow the title seperator to be changed dynamically.
		jQuery( '#title-separator input' ).on( 'click', autodescription.seperatorSwitch );

		// Initialise form field changing flag.
		autodescription.attachUnsavedChangesListener();

		// Initialize the status bar hover balloon.
		autodescription.statusBarHover();

		// Initialize status bar removal hover for touch screens.
		jQuery( document.body ).on( 'touchstart MSPointerDown', autodescription.removeDesc );

		// Bind reset confirmation
		jQuery( '.autodescription-js-confirm-reset' ).on( 'click.autodescription.autodescription_confirm_reset', autodescription.confirmReset );

		// Toggle Tabs in the SEO settings page.
		jQuery( '#social-tabs-js, #robots-tabs-js, #knowledge-tabs-js, #sitemaps-tabs-js' ).on( 'click', autodescription.tabToggle );

		// Toggle Title tagline aditions removal
		jQuery( '#title-tagline-toggle :input' ).on( 'click', autodescription.taglineToggle );
		// Check if it should be removed when page is loaded
		jQuery( document.body ).ready( autodescription.taglineToggleOnload );

		// Change Title Example prop on textual input change
		jQuery( '#autodescription-site-settings\\[homepage_title\\]' ).on( 'keyup', autodescription.titleProp );

	}

};

jQuery( autodescription.ready );
