<?php
/**
 * This file holds functions for easily extracting or interacting with data from The SEO FrameWork.
 *
 * @since 2.2.5
 *
 * We could bombard it with every public function, but that's very time consuming.
 * I'll add a bunch of functions on 2nd dot (v.X.v) release. e.g. 2.3.0, 2.4.0, etc.
 *
 * This will allow version comparing more easily (as you'll know how many users
 * use v.X version through the WordPress plugin stats.).
 * Therefor reducing work for you.
 */

/**
 * Load the class from cache.
 * This is recommended using this above using 'new The_SEO_Framework_Load();'
 * It also checks if the class is callable in the first place.
 *
 * @since 2.2.5
 */
function the_seo_framework() {
	return the_seo_framework_init();
}

/**
 * The SEO FrameWork version number
 *
 * Useful for version comparing
 *
 * @since 2.2.5
 *
 * @return string|null The SEO Framework three point version number. (e.g. '2.2.5')
 */
function the_seo_framework_version() {
	if ( the_seo_framework_active() )
		return THE_SEO_FRAMEWORK_VERSION;

	return;
}

/**
 * Check if The SEO FrameWork is active based on global filter.
 *
 * @since 2.2.5
 *
 * @return bool true if SEO framework is active
 */
function the_seo_framework_active() {
	return hmpl_ad_load();
}

/**
 * Compare the WordPress version to the input one.
 *
 * @since 2.2.9
 *
 * @param string $version The 3 point version compare
 * @param string $compare The PHP comparison operator.
 *
 * @return bool true if Version passes comparison.
 */
function tsf_wp_version( $version = '4.3.0', $compare = '>=' ) {
	//* We have to do it like so because we still need to support PHP 5.2 according to WP =/
	$theseoframework = the_seo_framework();

	return $theseoframework->wp_version( $version, $compare );
}

/**
 * Fetch the The SEO Framework Options pagehook.
 *
 * @since 2.2.9
 *
 * @return string The pagehook.
 */
function tsf_options_pagehook() {
	$theseoframework = the_seo_framework();

	return $theseoframework->pagehook;
}

/**
 * Fetch an option from The SEO Framework.
 *
 * @since 2.2.9
 *
 * @param string  $key       Option name.
 * @param boolean $use_cache Optional. Whether to use the cache value or not. Defaults to true.
 *
 * @return mixed The option value.
 */
function tsf_get_option( $key, $use_cache = true ) {
	$theseoframework = the_seo_framework();

	return $theseoframework->get_option( $key, $use_cache );
}
