<?php
/**
 * Class AutoDescription_Siteoptions
 *
 * Holds Site Options for the plugin.
 *
 * @since 2.2.2
 */
class AutoDescription_Siteoptions extends AutoDescription_Sanitize {

	/**
	 * Filterable Site Settings array.
	 *
	 * @since 2.2.2
	 *
	 * @var array Holds Site SEO options.
	 */
	protected $default_site_options = array();

	/**
	 * Site Settings field.
	 *
	 * @since 2.2.2
	 *
	 * @var string Settings field.
	 */
	protected $settings_field;

	/**
	 * Hold the Page ID for this class
	 *
	 * @since 2.2.2
	 *
	 * @var string The page ID
	 */
	protected $page_id;

	/**
	 * Constructor, load parent constructor and set up cachable variables.
	 */
	public function __construct() {
		parent::__construct();

		/**
		 * Default site settings. Seperated from Author, page or network settings.
		 *
		 * These settings can be overwritten per page or post depending on type and setting.
		 *
		 * @since 2.2.2
		 *
		 * No longer directly applies filters
		 * @since 2.2.7
		 */
		$this->default_site_options = array(
			'title_seperator'		=> 'pipe',	// Title seperator
			'title_location'		=> 'right',	// Title seperation location

			'noodp'					=> 1, 	// Site noopd robots settings
			'noydir'				=> 1, 	// Site noydir robots settings

			'category_noindex'		=> 0,	// Category Archive robots noindex
			'tag_noindex'			=> 0,	// Tag Archive robots noindex
			'author_noindex'		=> 0,	// Author Archive robots noindex
			'date_noindex'			=> 1,	// Date Archive robots noindex
			'search_noindex'		=> 1,	// Search Page robots noindex
			'site_noindex'			=> 0,	// Site Page robots noindex

			'category_nofollow'		=> 0,	// Category Archive robots nofollow
			'tag_nofollow'			=> 0,	// Tag Archive robots nofollow
			'author_nofollow'		=> 0,	// Author Archive robots nofollow
			'date_nofollow'			=> 0,	// Date Archive robots nofollow
			'search_nofollow'		=> 0,	// Search Page robots nofollow
			'site_nofollow'			=> 0,	// Site Page robots nofollow

			'category_noarchive'	=> 0,	// Category Archive robots noarchive
			'tag_noarchive'			=> 0,	// Tag Archive robots noarchive
			'author_noarchive'		=> 0,	// Author Archive robots noarchive
			'date_noarchive'		=> 0,	// Date Archive robots noarchive
			'search_noarchive'		=> 0,	// Search Page robots noarchive
			'site_noarchive'		=> 0,	// Site Page robots noarchive

			'paged_noindex'			=> 1,	// Every second or later page noindex

			'homepage_noindex'		=> 0,	// Home Page robots noindex
			'homepage_nofollow'		=> 0,	// Home Page robots noarchive
			'homepage_noarchive'	=> 0,	// Home Page robots nofollow

			'homepage_title'		=> '',	// Home Page Title string
			'homepage_tagline'		=> 1,	// Home Page add blog Tagline
			'homepage_description'	=> '',	// Home Page Description string
			'home_title_location'	=> 'left',	// Title seperation location

			'shortlink_tag'			=> 0,	// Adds shortlink tag

			'prev_next_posts'		=> 0,	// Adds next/prev tags
			'prev_next_archives'	=> 1,	// Adds next/prev tags

			'facebook_publisher'	=> '',	// Facebook Business Url
			'facebook_author'		=> '',	// Facebook User URl
			'facebook_appid'		=> '',	// Facebook App ID

			'post_publish_time'		=> 1,	// Article Published Time
			'post_modify_time'		=> 1,	// Article Modified Time

			'page_publish_time'		=> 0,	// Article Published Time
			'page_modify_time'		=> 0,	// Article Modified Time

			'home_publish_time'		=> 0,	// Article Modified Time
			'home_modify_time'		=> 0,	// Article Modified Time

			'twitter_card' 			=> 'summary_large_image',	// Twitter Card layout. If no twitter:image image is found, it'll change to 'summary'
			'twitter_site' 			=> '', 	// Twitter business @username
			'twitter_creator' 		=> '', 	// Twitter user @username

			'og_tags' 				=> 1,	// Output of Open Graph meta tags
			'facebook_tags'			=> 0, 	// Output the Facebook meta tags
			'twitter_tags'			=> 0, 	// Output the Twitter meta tags

			'google_verification'	=> '', 	// Google Verification Code
			'bing_verification'		=> '', 	// Bing Verification Code

			// https://developers.google.com/structured-data/customize/contact-points - This is extremely extended and valuable. Expect a premium version.
			'knowledge_output'		=> 1,				// Default for outputing the Knowledge SEO.
			'knowledge_type'		=> 'organization',	// Organization or Person

			// https://developers.google.com/structured-data/customize/logos
			'knowledge_logo'		=> 1,	// Fetch logo from WP Favicon
			'knowledge_name'		=> '',	// Person or Organization name

			// 'Sameas'
			'knowledge_facebook'	=> '',	// Facebook Account
			'knowledge_twitter'		=> '',	// Twitter Account
			'knowledge_gplus'		=> '',	// Google Plus Account
			'knowledge_instagram'	=> '',	// Instagram Account
			'knowledge_youtube'		=> '',	// Youtube Account
			'knowledge_linkedin'	=> '',	// Linkedin Account
		//	'knowledge_myspace'		=> '',	// MySpace Account // meh.
			'knowledge_pinterest'	=> '',	// Pinterest Account
			'knowledge_soundcloud'	=> '',	// SoundCloud Account
			'knowledge_tumblr'		=> '',	// Tumblr Account

			// Sitemaps
			'sitemaps_output'		=> 1,	// Output of sitemaps
			'sitemaps_modified'		=> 1,	// Add sitemaps modified time
			'sitemaps_robots'		=> 1,	// Add sitemaps location to robots.txt
			'ping_google'			=> 1,	// Ping Google
			'ping_bing'				=> 1,	// Ping Bing
			'ping_yahoo'			=> 1,	// Ping Yahoo
		);

		$this->settings_field = THE_SEO_FRAMEWORK_SITE_OPTIONS;

		//* Set up site settings and save/reset them
		add_action( 'admin_init', array( $this, 'register_settings' ) );

		// The page_id
		$this->page_id = 'autodescription-settings';

		/**
		 * Add plugin links to the plugin activation page.
		 * @since 2.2.8
		 */
		add_filter( 'plugin_action_links_' . THE_SEO_FRAMEWORK_PLUGIN_BASENAME, array( $this, 'plugin_action_links' ), 10, 2 );
	}

	/**
	 * Return the compiled options.
	 *
	 * @since 2.2.7
	 *
	 * @param array $args The new options through filter.
	 * @return array The SEO Framework Options
	 */
	 protected function default_site_options( $args = array() ) {

		/* Marked as active @since 2.3.5
		return $this->default_site_options = wp_parse_args(
			$args,
			apply_filters(
				'the_seo_framework_default_site_options',
				wp_parse_args(
					$args,
					$this->default_site_options
				)
			)
		);*/

		/**
		 * Filter The SEO Framework options.
		 *
		 * @since 2.2.7
		 */
		//* Marked for removal @since 2.3.5
		$defaults = apply_filters( 'autodescription_default_site_options',
			wp_parse_args(
				$args,
				$this->default_site_options
			)
		);

		//* @since 2.3.0 filter.
		$defaults = (array) apply_filters( 'the_seo_framework_default_site_options', $defaults );

		return $this->default_site_options = wp_parse_args(
			$args,
			$defaults
		);

	}

	/**
	 * Register meta boxes on the Site SEO Settings page.
	 *
	 * @since 2.2.2
	 *
	 * @see $this->title_metabox()		Callback for Title Settings box.
	 * @see $this->robots_metabox()		Callback for Robots Settings box.
	 * @see $this->homepage_metabox()	Callback for Home Page Settings box.
	 * @see $this->social_metabox()		Callback for Social Settings box.
	 * @see $this->webmaster_metabox()	Callback for Webmaster Settings box.
	 */
	public function metaboxes() {

		/**
		 * Various metabox filters.
		 * Set any to false if you wish the meta box to be removed.
		 *
		 * @since 2.2.4
		 */
		//* Marked for removal @since 2.3.5
		$title = apply_filters( 'autodescription_title_metabox', true );
		//* @since 2.3.0 filter.
		$title = (bool) apply_filters( 'the_seo_framework_title_metabox', $title );

		//* Marked for removal @since 2.3.5
		$robots = apply_filters( 'autodescription_robots_metabox', true );
		//* @since 2.3.0 filter.
		$robots = (bool) apply_filters( 'the_seo_framework_robots_metabox', $robots );

		//* Marked for removal @since 2.3.5
		$home = apply_filters( 'autodescription_home_metabox', true );
		//* @since 2.3.0 filter.
		$home = (bool) apply_filters( 'the_seo_framework_home_metabox', $home );

		//* Marked for removal @since 2.3.5
		$social = apply_filters( 'autodescription_social_metabox', true );
		//* @since 2.3.0 filter.
		$social = (bool) apply_filters( 'the_seo_framework_social_metabox', $social );

		//* Marked for removal @since 2.3.5
		$knowledge = apply_filters( 'autodescription_knowledge_metabox', true );
		//* @since 2.3.0 filter.
		$knowledge = (bool) apply_filters( 'the_seo_framework_knowledge_metabox', $knowledge );

		//* Marked for removal @since 2.3.5
		$webmaster = apply_filters( 'autodescription_webmaster_metabox', true );
		//* @since 2.3.0 filter.
		$webmaster = (bool) apply_filters( 'the_seo_framework_webmaster_metabox', $webmaster );

		//* Marked for removal @since 2.3.5
		$sitemap = apply_filters( 'autodescription_sitemap_metabox', true );
		//* @since 2.3.0 filter.
		$sitemap = (bool) apply_filters( 'the_seo_framework_sitemap_metabox', $sitemap );

		//* Title Meta Box
		if ( $title )
			add_meta_box(
				'autodescription-title-settings',
				__( 'Title Settings', 'autodescription' ),
				array( $this, 'title_metabox' ),
				$this->pagehook,
				'main'
			);

		//* Home Page Meta Box
		if ( $home )
			add_meta_box(
				'autodescription-homepage-settings',
				__( 'Home Page Settings', 'autodescription' ),
				array( $this, 'homepage_metabox' ),
				$this->pagehook,
				'main'
			);

		//* Social Meta Box
		if ( $social )
			add_meta_box(
				'autodescription-social-settings',
				__( 'Social Meta Settings', 'autodescription' ),
				array( $this, 'social_metabox' ),
				$this->pagehook,
				'main'
			);

		//* Knowledge Graph Meta Box
		if ( $knowledge )
			add_meta_box(
				'autodescription-knowledgegraph-settings',
				__( 'Knowledge Graph Settings', 'autodescription' ),
				array( $this, 'knowledge_metabox' ),
				$this->pagehook,
				'main'
			);

		//* Robots Meta Box
		if ( $robots )
			add_meta_box(
				'autodescription-robots-settings',
				__( 'Robots Meta Settings', 'autodescription' ),
				array( $this, 'robots_metabox' ),
				$this->pagehook,
				'main'
			);

		//* Webmaster Meta Box
		if ( $webmaster )
			add_meta_box(
				'autodescription-webmaster-settings',
				__( 'Webmaster Meta Settings', 'autodescription' ),
				array( $this, 'webmaster_metabox' ),
				$this->pagehook,
				'main'
			);

		//* Sitemaps Meta Box
		if ( $sitemap )
			add_meta_box(
				'autodescription-sitemap-settings',
				__( 'Sitemaps Settings', 'autodescription' ),
				array( $this, 'sitemaps_metabox' ),
				$this->pagehook,
				'main'
			);

	}

	/**
	 * Register the database settings for storage.
	 *
	 * @since 2.2.2
	 *
	 * @return void
	 *
	 * @thanks StudioPress
	 */
	public function register_settings() {
		//* If this page doesn't store settings, no need to register them
		if ( ! $this->settings_field )
			return;

		register_setting( $this->settings_field, $this->settings_field );
		add_option( $this->settings_field, $this->default_site_options() );

		//* If this page isn't the SEO Settings page, there's no need to check for a reset.
		if ( ! $this->is_menu_page( $this->page_id ) )
			return;

		if ( $this->get_option( 'reset', $this->settings_field ) ) {
			if ( update_option( $this->settings_field, $this->default_site_options() ) )
				$this->admin_redirect( $this->page_id, array( 'reset' => 'true' ) );
			else
				$this->admin_redirect( $this->page_id, array( 'error' => 'true' ) );
			exit;
		}

	}

	/**
	 * Get the default of any of the The SEO Framework settings.
	 *
	 * @since 2.2.4
	 *
	 * @uses $this->settings_field
	 * @uses $this->default_site_options()
	 *
	 * @param string $key required The option name
	 * @param string $setting optional The settings field
	 * @param bool $use_cache optional Use the options cache or not. For debugging purposes.
	 *
	 * @return 	int|bool|string default option
	 *			int '-1' if option doesn't exist.
	 */
	public function get_default_settings( $key, $setting = '', $use_cache = true ) {

		if ( empty( $key ) )
			return false;

		// Unused, for now.
		if ( empty( $setting ) )
			$setting = $this->settings_field;

		//* If we need to bypass the cache
		if ( ! $use_cache ) {
			$defaults = $this->default_site_options();

			if ( ! is_array( $defaults ) || ! array_key_exists( $key, $defaults ) )
				return -1;

			return is_array( $defaults[$key] ) ? stripslashes_deep( $defaults[$key] ) : stripslashes( wp_kses_decode_entities( $defaults[$key] ) );
		}

		static $defaults_cache = array();

		//* Check options cache
		if ( isset( $defaults_cache[$key] ) )
			//* Option has been cached
			return $defaults_cache[$key];

		$defaults_cache = $this->default_site_options();

		if ( ! is_array( $defaults_cache ) || ! array_key_exists( $key, (array) $defaults_cache ) )
			$defaults_cache[$key] = -1;

		return $defaults_cache[$key];
	}


	/**
	 * Adds link from plugins page to SEO Settings page.
	 *
	 * @since 2.2.8
	 */
	public function plugin_action_links( $links ) {

		$framework_links = array(
			'settings' => '<a href="' . esc_url( admin_url( 'admin.php?page=' . $this->page_id ) ) . '">' . __( 'SEO Settings', 'autodescription' ) . '</a>',
			'home' => '<a href="'. esc_url( 'http://theseoframework.com', 'http' ) . '" target="_blank">' . _x( 'Plugin Home', 'As in: The Plugin Home Page', 'autodescription' ) . '</a>'
		);

		return array_merge( $framework_links, $links );
	}

}
