<?php
/**
 * Class AutoDescription_Render
 *
 * Puts all data into HTML valid strings
 * Returns strings
 *
 * @since 2.1.6
 */
class AutoDescription_Render extends AutoDescription_Admin_Init {

	/**
	 * Constructor, load parent constructor
	 */
	public function __construct() {
		parent::__construct();
	}

	/**
	 * Cache description in static variable
	 * Must be called inside the loop
	 *
	 * @since 2.2.2
	 * @return string The description
	 */
	public function description_from_cache() {

		static $description_cache = null;

		if ( isset( $description_cache ) )
			return $description_cache;

		$description_cache = $this->generate_description();

		return $description_cache;
	}

	/**
	 * Cache current URL in static variable
	 * Must be called inside the loop
	 *
	 * @param string $url the url
	 * @param int $page_id the page id, if empty it will fetch the requested ID, else the page uri
	 * @param bool $paged Return current page URL without pagination
	 * @param bool $from_option Get the canonical uri option
	 *
	 * @since 2.2.2
	 * @return string The url
	 */
	public function the_url_from_cache( $url = '', $page_id = '', $paged = false, $from_option = true ) {

		static $url_cache = array();

		if ( isset( $url_cache[$url][$page_id][$paged][$from_option] ) )
			return $url_cache[$url][$page_id][$paged][$from_option];

		$url_cache[$url][$page_id][$paged][$from_option] = $this->the_url( $url, $page_id, $paged, $from_option );

		return $url_cache[$url][$page_id][$paged][$from_option];
	}

	/**
	 * Cache current Title in static variable
	 * Must be called inside the loop
	 *
	 * @param string $title The Title to return
	 * @param string $sep The Title sepeartor
	 * @param string $seplocation The Title sepeartor location ( accepts 'left' or 'right' )
	 * @param bool $meta Continue normally if set to true, regardless of theme doing it right.
	 *
	 * @since 2.2.2
	 * @return string The title
	 */
	public function title_from_cache( $title = '', $sep = '', $seplocation = '', $meta = false ) {

		static $title_cache = array();

		if ( isset( $title_cache[$title][$sep][$seplocation][$meta] ) )
			return $title_cache[$title][$sep][$seplocation][$meta];

		$title_cache[$title][$sep][$seplocation][$meta] = $this->title( $title, $sep, $seplocation, '', '', '', '', '', $meta );

		return $title_cache[$title][$sep][$seplocation][$meta];
	}

	/**
	 * Cache current Image URL in static variable
	 * Must be called inside the loop
	 *
	 * @since 2.2.2
	 * @return string The image url
	 */
	public function get_image_from_cache() {

		static $image_cache = null;

		if ( isset( $image_cache ) )
			return $image_cache;

		$post_id = get_the_ID();

		//* End this madness if there's no ID found (search/404/etc.)
		if ( ! $post_id )
			return '';

		$image_cache = esc_url_raw( $this->get_image( $post_id ) );

		return $image_cache;
	}

	/**
	 * Render the description
	 *
	 * @uses $this->description_from_cache()
	 * @uses $this->detect_seo_plugins()
	 *
	 * @since 1.3.0
	 */
	public function the_description() {

		if ( $this->detect_seo_plugins() )
			return;

		//* @since 2.3.0
		$description = (string) apply_filters( 'the_seo_framework_description_output', '' );

		if ( empty( $description ) )
			$description = $this->description_from_cache();

		return '<meta name="description" content="' . esc_attr( $description ) . '" />' . "\r\n";
	}

	/**
	 * Render og:description
	 *
	 * @uses $this->description_from_cache()
	 * @uses $this->has_og_plugin()
	 *
	 * @since 1.3.0
	 */
	public function og_description() {

		if ( $this->has_og_plugin() !== false )
			return;

		/**
		 * Return if OG tags are disabled
		 *
		 * @since 2.2.2
		 */
		if ( ! $this->get_option( 'og_tags' ) )
			return;

		//* @since 2.3.0
		$description = (string) apply_filters( 'the_seo_framework_ogdescription_output', '' );

		if ( empty( $description ) )
			$description = $this->description_from_cache();

		return '<meta property="og:description" content="' . esc_attr( $description ) . '" />' . "\r\n";
	}

	/**
	 * Render the locale
	 *
	 * @uses $this->has_og_plugin()
	 *
	 * @since 1.0.0
	 */
	public function og_locale() {

		if ( $this->has_og_plugin() !== false )
			return;

		/**
		 * Return if OG tags are disabled
		 *
		 * @since 2.2.2
		 */
		if ( ! $this->get_option( 'og_tags' ) )
			return;

		//* @since 2.3.0
		$locale = (string) apply_filters( 'the_seo_framework_oglocale_output', '' );

		if ( empty( $locale ) )
			$locale = get_locale();

		return '<meta property="og:locale" content="' . esc_attr( $locale ) . '" />' . "\r\n";
	}

	/**
	 * Process the title to WordPress
	 *
	 * @uses $this->title_from_cache()
	 * @uses $this->has_og_plugin()
	 *
	 * @since 2.0.3
	 */
	public function og_title() {

		if ( $this->has_og_plugin() !== false )
			return;

		/**
		 * Return if OG tags are disabled
		 *
		 * @since 2.2.2
		 */
		if ( ! $this->get_option( 'og_tags' ) )
			return;

		//* @since 2.3.0
		$title = (string) apply_filters( 'the_seo_framework_ogtitle_output', '' );

		if ( empty( $title ) )
			$title = $this->title_from_cache( '', '', '', true );

		return '<meta property="og:title" content="' . esc_attr( $title ) . '" />' . "\r\n";
	}

	/**
	 * Get the type
	 *
	 * @uses $this->has_og_plugin()
	 *
	 * @since 1.1.0
	 */
	public function og_type() {

		if ( $this->has_og_plugin() !== false )
			return;

		/**
		 * Return if OG tags are disabled
		 *
		 * @since 2.2.2
		 */
		if ( ! $this->get_option( 'og_tags' ) )
			return;

		//* @since 2.3.0
		$type = (string) apply_filters( 'the_seo_framework_ogtype_output', '' );

		if ( empty( $type ) ) {
			$type = 'website';

			if ( is_single() ) {
				$type = 'article';
			}

			if ( is_front_page() || is_page() ) {
				$type = 'website';
			}

			if ( is_author() ) {
				$type = 'profile';
			}
		}

		return '<meta property="og:type" content="' . esc_attr( $type ) . '" />' . "\r\n";
	}

	/**
	 * Adds og:image
	 *
	 * @param string $image url for image
	 *
	 * @since 1.3.0
	 */
	public function og_image() {

		/**
		 * Return if OG tags are disabled
		 *
		 * @since 2.2.2
		 */
		if ( ! $this->get_option( 'og_tags' ) )
			return;

		//* @since 2.3.0
		$image = (string) apply_filters( 'the_seo_framework_ogimage_output', '' );

		if ( empty( $image ) )
			$image = $this->get_image_from_cache();

		if ( empty( $image ) )
			$image = $image;

		/**
		 * Always output
		 *
		 * @since 2.1.1
		 */
		return '<meta property="og:image" content="' . esc_attr( $image ) . '" />' . "\r\n";
	}

	/**
	 * Adds og:site_name
	 *
	 * @uses wp
	 *
	 * @param string output	the output
	 *
	 * @since 1.3.0
	 */
	public function og_sitename() {

		//* if WPSEO is active
		if ( $this->has_og_plugin() !== false )
			return;

		/**
		 * Return if OG tags are disabled
		 *
		 * @since 2.2.2
		 */
		if ( ! $this->get_option( 'og_tags' ) )
			return;

		//* @since 2.3.0
		$sitename = (string) apply_filters( 'the_seo_framework_ogsitename_output', '' );

		if ( empty( $sitename ) )
			$sitename = get_bloginfo('name');

		return '<meta property="og:site_name" content="' . esc_attr( $sitename ) . '" />' . "\r\n";
	}

	/**
	 * Adds og:url
	 *
	 * @return string og:url the url meta
	 *
	 * @since 1.3.0
	 *
	 * @uses $this->the_url_from_cache()
	 */
	public function og_url() {

		//* if WPSEO is active
		if ( $this->has_og_plugin() !== false )
			return;

		/**
		 * Return if OG tags are disabled
		 *
		 * @since 2.2.2
		 */
		if ( ! $this->get_option( 'og_tags' ) )
			return;

		return '<meta property="og:url" content="' . esc_attr( $this->the_url_from_cache() ) . '" />' . "\r\n";
	}

	/**
	 * Render twitter:card
	 *
	 * @uses $this->generate_description()
	 * @uses $this->has_og_plugin()
	 *
	 * @since 2.2.2
	 */
	public function twitter_card() {

		if ( ! $this->get_option( 'twitter_tags' ) )
			return;

		//* @since 2.3.0
		$card = (string) apply_filters( 'the_seo_framework_twittercard_output', '' );

		if ( empty( $card ) ) {
			$card = $this->get_option( 'twitter_card' );

			/**
			 * Return card type if image is found
			 * Return to summary if not
			 */
			if ( ! $this->get_image_from_cache() ) {
				$card = 'summary';
			}
		}

		return '<meta name="twitter:card" content="' . esc_attr( $card ) . '" />' . "\r\n";
	}

	/**
	 * Render twitter:site
	 *
	 * @since 2.2.2
	 */
	public function twitter_site() {

		if ( ! $this->get_option( 'twitter_tags' ) )
			return;

		//* @since 2.3.0
		$site = (string) apply_filters( 'the_seo_framework_twittersite_output', '' );

		if ( empty( $site ) ) {
			$site = $this->get_option( 'twitter_site' );

			/**
			 * Return empty if no twitter_site is found
			 */
			if ( empty( $site ) )
				return '';
		}

		return '<meta name="twitter:site" content="' . esc_attr( $site ) . '" />' . "\r\n";
	}

	/**
	 * Render twitter:creator or twitter:site:id
	 *
	 * @since 2.2.2
	 */
	public function twitter_creator() {

		if ( ! $this->get_option( 'twitter_tags' ) )
			return;

		//* @since 2.3.0
		$creator = (string) apply_filters( 'the_seo_framework_twittercreator_output', '' );

		if ( empty( $creator ) ) {
			$site = $this->get_option( 'twitter_site' );
			$creator = $this->get_option( 'twitter_creator' );

			/**
			 * Return site:id instead of creator is no twitter:site is found.
			 * Per Twitter requirements
			 */
			if ( empty( $site ) && !empty( $creator ) )
				return '<meta name="twitter:site:id" content="' . esc_attr( $creator ) . '" />' . "\r\n";
		}

		if ( empty( $creator ) )
			return '';

		return '<meta name="twitter:creator" content="' . esc_attr( $creator ) . '" />' . "\r\n";
	}

	/**
	 * Render twitter:title
	 *
	 * @uses $this->title_from_cache()
	 * @uses $this->has_og_plugin()
	 *
	 * @since 2.2.2
	 */
	public function twitter_title() {

		if ( ! $this->get_option( 'twitter_tags' ) )
			return;

		//* @since 2.3.0
		$title = (string) apply_filters( 'the_seo_framework_twittertitle_output', '' );

		if ( empty( $title ) )
			$title = $this->title_from_cache( '', '', '', true );

		return '<meta name="twitter:title" content="' . esc_attr( $title ) . '" />' . "\r\n";
	}

	/**
	 * Render twitter:description
	 *
	 * @uses $this->description_from_cache()
	 *
	 * @since 2.2.2
	 */
	public function twitter_description() {

		if ( ! $this->get_option( 'twitter_tags' ) )
			return;

		//* @since 2.3.0
		$description = (string) apply_filters( 'the_seo_framework_twitterdescription_output', '' );

		if ( empty( $description ) )
			$description = $this->description_from_cache();

		return '<meta name="twitter:description" content="' . esc_attr( $description ) . '" />' . "\r\n";
	}

	/**
	 * Render twitter:image
	 *
	 * @param string $image url for image
	 *
	 * @since 2.2.2
	 *
	 * @return string|null The twitter image source meta tag
	 */
	public function twitter_image() {

		if ( ! $this->get_option( 'twitter_tags' ) )
			return;

		//* @since 2.3.0
		$image = (string) apply_filters( 'the_seo_framework_twitterimage_output', '' );

		if ( empty( $image ) )
			$image = $this->get_image_from_cache();

		if ( !empty( $image ) ) {
			return '<meta name="twitter:image:src" content="' . esc_attr( $image ) . '" />' . "\r\n";
		} else {
			return '';
		}

	}

	/**
	 * Render article:author
	 *
	 * @since 2.2.2
	 *
	 * @return string|null The facebook app id
	 */
	public function facebook_author() {

		if ( ! $this->get_option( 'facebook_tags' ) )
			return;

		//* @since 2.3.0
		$author = (string) apply_filters( 'the_seo_framework_facebookauthor_output', '' );

		if ( empty( $author ) )
			$author = $this->get_option( 'facebook_author' );

		if ( ! empty( $author ) )
			return '<meta name="article:author" content="' . esc_attr( esc_url_raw( $author ) ) . '" />' . "\r\n";

		return '';
	}

	/**
	 * Render article:author
	 *
	 * @since 2.2.2
	 *
	 * @return string|null The facebook app id
	 */
	public function facebook_publisher() {

		if ( ! $this->get_option( 'facebook_tags' ) )
			return;

		//* @since 2.3.0
		$publisher = (string) apply_filters( 'the_seo_framework_facebookpublisher_output', '' );

		if ( empty( $publisher ) )
			$publisher = $this->get_option( 'facebook_publisher' );

		if ( ! empty( $publisher ) )
			return '<meta name="article:publisher" content="' . esc_attr( esc_url_raw( $publisher ) ) . '" />' . "\r\n";

		return '';
	}

	/**
	 * Render fb:app_id
	 *
	 * @since 2.2.2
	 *
	 * @return string|null The facebook app id
	 */
	public function facebook_app_id() {

		if ( ! $this->get_option( 'facebook_tags' ) )
			return;

		//* @since 2.3.0
		$app_id = (string) apply_filters( 'the_seo_framework_facebookappid_output', '' );

		if ( empty( $app_id ) )
			$app_id = $this->get_option( 'facebook_appid' );

		if ( ! empty( $app_id ) )
			return '<meta name="fb:app_id" content="' . esc_attr( $app_id ) . '" />' . "\r\n";

		return '';
	}

	/**
	 * Render article:published_time
	 *
	 * @since 2.2.2
	 *
	 * @return string|null The article:published_time
	 */
	public function article_published_time() {

		// Don't do anything if it's not a page or post.
		if ( !is_singular() )
			return;

		// If it's a post, but the option is disabled, don't do anyhting.
		if ( !is_front_page() && is_single() && !$this->get_option( 'post_publish_time' ) )
			return;

		// If it's a page, but the option is disabled, don't do anything.
		if ( !is_front_page() && is_page() && !$this->get_option( 'page_publish_time' ) )
			return;

		// If it's  the home page, but the option is disabled, don't do anything.
		if ( is_front_page() && !$this->get_option( 'home_publish_time' ) )
			return;

		//* @since 2.3.0
		$time = (string) apply_filters( 'the_seo_framework_publishedtime_output', '' );

		if ( empty( $time ) )
			$time = get_the_date( 'Y-m-d', '' );

		if ( ! empty( $time ) )
			return '<meta name="article:published_time" content="' . esc_attr( $time ) . '" />' . "\r\n";

		return '';
	}

	/**
	 * Render article:modified_time
	 *
	 * @since 2.2.2
	 *
	 * @return string|null The article:modified_time
	 */
	public function article_modified_time() {

		// Don't do anything if it's not a page or post, or if both options are disabled
		if ( !is_singular() )
			return;

		// If it's a post, but the option is disabled, don't do anyhting.
		if ( !is_front_page() && is_single() && ! $this->get_option( 'post_modify_time' ) )
			return;

		// If it's a page, but the option is disabled, don't do anything.
		if ( !is_front_page() && is_page() && ! $this->get_option( 'page_modify_time' ) )
			return;

		// If it's the home page, but the option is disabled, don't do anything.
		if ( is_front_page() && ! $this->get_option( 'home_modify_time' ) )
			return;

		//* @since 2.3.0
		$time = (string) apply_filters( 'the_seo_framework_modifiedtime_output', '' );

		if ( empty( $time ) )
			$time = the_modified_date( 'Y-m-d', '', '', false );

		if ( ! empty( $time ) ) {
			$output = '<meta name="article:modified_time" content="' . esc_attr( $time ) . '" />' . "\r\n";

			if ( $this->get_option( 'og_tags' ) )
				$output .= '<meta property="og:updated_time" content="' . esc_attr( $time ) . '" />'. "\r\n";

			return $output;
		}

		return '';
	}

	/**
	 * Outputs canonical url
	 *
	 * @since 2.0.6
	 *
	 * @uses $this->the_url_from_cache()
	 *
	 * @return string canonical url meta
	 */
	public function canonical() {

		//* if WPSEO is active
		if ( $this->has_og_plugin() !== false )
			return;

		if ( ! get_option( 'permalink_structure' ) )
			return;

		return '<link rel="canonical" href="' . esc_attr( $this->the_url_from_cache() ) . '" />' . "\r\n";
	}

	/**
	 * LD+JSON helper output
	 *
	 * @uses $this->has_json_ld_plugin()
	 * @uses $this->ld_json_search()
	 * @uses $this->ld_json_knowledge()
	 *
	 * @since 1.2.0
	 * @return string $output LD+json helpers in header on front page.
	 */
	public function ld_json() {

		//* Check for WPSEO LD+JSON
		if ( $this->has_json_ld_plugin() !== false )
			return;

		//* Only display on front page
		if ( !is_front_page() )
			return;

		$output = '';

		/**
		 * Add multiple scripts
		 *
		 * @since 2.2.8
		 */
		$searchhelper = $this->ld_json_search();
		$knowledgegraph = $this->ld_json_knowledge();

		if ( !empty( $searchhelper ) )
			$output .= "<script type='application/ld+json'>" . $searchhelper . "</script>" . "\r\n";

		if ( !empty( $knowledgegraph ) )
			$output .= "<script type='application/ld+json'>" . $knowledgegraph . "</script>" . "\r\n";

		return $output;
	}

	/**
	 * Outputs Google Site Verification code
	 *
	 * @since 2.2.4
	 *
	 * @return string|null google verification code
	 */
	public function google_site_output() {

		//* @since 2.3.0
		$code = (string) apply_filters( 'the_seo_framework_googlesite_output', '' );

		if ( empty( $code ) )
			$code = $this->get_option( 'google_verification' );

		if ( empty( $code ) )
			return '';

		return '<meta name="google-site-verification" content="' . esc_attr( $code ) . '" />' . "\r\n";
	}

	/**
	 * Outputs Bing Site Verification code
	 *
	 * @since 2.2.4
	 *
	 * @return string|null Bing Webmaster code
	 */
	public function bing_site_output() {

		//* @since 2.3.0
		$code = (string) apply_filters( 'the_seo_framework_bingsite_output', '' );

		if ( empty( $code ) )
			$code = $this->get_option( 'bing_verification' );

		if ( empty( $code ) )
			return '';

		return '<meta name="msvalidate.01" content="' . esc_attr( $code ) . '" />' . "\r\n";
	}

	/**
	 * Output the `index`, `follow`, `noodp`, `noydir`, `noarchive` robots meta code in the document `head`.
	 *
	 * @since 2.0.0
	 *
	 * @return null Return early if blog is not public.
	 */
	public function robots() {

		// Don't do anything if the blog isn't set to public.
		if ( ! get_option( 'blog_public' ) )
			return '';

		$robots = '';
		$meta = $this->robots_meta();

		//* Add meta if any exist
		if ( $meta )
			return sprintf( '<meta name="robots" content="%s" />' . "\r\n", implode( ',', $meta ) );

		 return '';
	}

	/**
	 * Outputs favicon urls
	 *
	 * @since 2.2.1
	 *
	 * @uses $this->site_icon()
	 *
	 * @return string icon links.
	 * @TODO Make this work for older wp versions. i.e. add upload area for wp 4.2.99999 and lower
	 * @TODO Make this work in the first place
	 */
	public function favicon() {

		if ( $this->wp_version( '4.3.0', '<' ) ) {
			$output = '<link rel="icon" type="image/x-icon" href="' . esc_url( $this->site_icon( 16 ) ) . '" sizes="16x16" />' . "\r\n";
			$output .= '<link rel="icon" type="image/x-icon" href="' . esc_url( $this->site_icon( 192 ) ) . '" sizes="192x192" />' . "\r\n";
			$output .= '<link rel="apple-touch-icon-precomposed" href="' . esc_url( $this->site_icon( 180 ) ) . '" />' . "\r\n";
			$output .= '<link rel="msapplication-TileImage" href="' . esc_url( $this->site_icon( 270 ) ) . '" />' . "\r\n";

			return $output;
		}

		return '';
	}

	/**
	 * Outputs shortlink meta tag
	 *
	 * @since 2.2.2
	 *
	 * @uses $this->get_shortlink()
	 *
	 * @return string|null shortlink url meta
	 */
	public function shortlink() {

		$url = $this->get_shortlink();

		if ( $url )
			return sprintf( '<link rel="shortlink" href="%s" />' . "\r\n", $url );

		return '';
	}

	/**
	 * Outputs shortlink meta tag
	 *
	 * @since 2.2.2
	 *
	 * @uses $this->get_shortlink()
	 *
	 * @return string|null shortlink url meta
	 */
	public function paged_urls() {

		$next = $this->get_paged_url( 'next' );
		$prev = $this->get_paged_url( 'prev' );

		$output = '';

		if ( $prev )
			$output .= sprintf( '<link rel="prev" href="%s" />' . "\r\n", $prev );
		if ( $next )
			$output .= sprintf( '<link rel="next" href="%s" />' . "\r\n", $next );

		return $output;
	}

}
