<?php
/**
 * Class AutoDescription_Inpost
 *
 * Outputs Taxonomy, Post and Page meta boxes
 *
 * @since 2.2.2
 */
class AutoDescription_Inpost extends AutoDescription_PageOptions {

	/**
	 * Constructor, load parent constructor
	 */
	public function __construct() {
		parent::__construct();

		// Enqueue inpost meta boxes
		add_action( 'add_meta_boxes', array( $this, 'add_inpost_seo_box_init' ), 5 );

		// Enqueue taxonomy meta boxes
		add_action( 'admin_init', array( $this, 'add_taxonomy_seo_box_init' ), 9 );
	}

	/**
	 * Render the SEO meta box
	 *
	 * Called outside autodescription_run
	 *
	 * Applies hmpl_ad_seobox filters. Return false to disable the meta boxes
	 *
	 * @since 2.0.0
	 */
	public function add_inpost_seo_box_init() {
		if ( $this->detect_seo_plugins() )
			return '';

		//* Marked for removal @since 2.3.5
		$hmpl_ad_seobox = apply_filters( 'hmpl_ad_seobox', true );
		//* @since 2.3.0 filter.
		$hmpl_ad_seobox = (bool) apply_filters( 'the_seo_framework_seobox_output', $hmpl_ad_seobox );

		if ( $hmpl_ad_seobox !== false )
			add_action( 'add_meta_boxes', array( $this, 'add_inpost_seo_box' ), 10 );
	}

	/**
	 * Adds SEO Meta boxes within Taxonomy screens
	 *
	 * @since 2.1.8
	 *
	 * @options Genesis : Merge these options with Genesis options. Prevents lost data.
	 */
	public function add_taxonomy_seo_box_init() {
		// Add taxonomy meta boxes
		foreach ( get_taxonomies( array( 'public' => true ) ) as $tax_name )
			add_action( $tax_name . '_edit_form', array( &$this, 'pre_seo_box'), 10, 2 );
	}

	/**
	 * Adds SEO Meta boxes beneath every page/post edit screen
	 *
	 * High priority, this box is seen right below the post/page edit screen.
	 *
	 * @since 2.0.0
	 *
	 * @options Genesis : Merge these options with Genesis options. Prevents lost data.
	 */
	public function add_inpost_seo_box() {

		$post_page = (array) get_post_types( array( 'public' => true ) );

		//* Adds meta boxes on Posts/Pages
		foreach ( $post_page as $screen ) {

			$post = __( 'Page', 'autodescription' );

			if ( $screen == 'post' ) {
				$post = __( 'Post', 'autodescription' );
				$title = $post;
			} else if ( $screen == 'page' ) {
				$title = $post;
			} else {
				$title = __( 'Theme', 'autodescription' );
			}

			$args = array( $post, 'is_post_page' );

			$id = 'autodescription-seo-box';

			if ( post_type_supports( $screen, 'autodescription-meta' ) ) {
				// Note: Pass on the object $this
				add_meta_box( $id, sprintf( __( '%s SEO Settings', 'autodescription' ), $title ), array( &$this, 'pre_seo_box' ), $screen, 'normal', 'high', $args );
			}
		}

	}

	/**
	 * Determines which arguments should be used
	 *
	 * @since 2.1.8
	 *
	 * @used by add_inpost_seo_box
	 *
	 * @param $object the page/post/taxonomy object
	 * @param $args the page/post arguments or taxonomy slug
	 */
	public function pre_seo_box( $object, $args ) {

		if ( is_array( $args ) && isset( $args['args'] ) ) {
			$args_split = $args['args'];

			$page = $args_split[1];

			// Return $args as array on post/page
			if ( $page === 'is_post_page') {
				// Note: Passes through object.
				return $this->inpost_seo_box( $object, (array) $args );
			}
		} else {
			// Note: Passes through object.
			// Empty the arguments, if any.
			return $this->inpost_seo_box( $object, $args = '' );
		}
	}

	/**
	 * Callback for in-post SEO meta box.
	 *
	 * @since 2.0.0
	 *
	 * @param array $post		The post object
	 *
	 * @param object $object 	the page/post/taxonomy object
	 * @param array $args 		the page/post arguments or taxonomy slug
	 *
	 * @uses $this->get_custom_field() Get custom field value.
	 *
	 * Note: Passed through object $object by reference
	 */
	public function inpost_seo_box( &$object, $args ) {

		//* Determines if it's inside a meta box or within a taxonomy page.
		$nobox = false;

		// Args are passed.
		if ( is_array( $args ) && isset( $args['args'] ) ) {
			$args_split = $args['args'];

			//* The post type callback arg (translated)
			$type = $args_split[0];
			//* The kind of page we're on.
			$page = $args_split[1];

			// Only add nonce on post/page edit screen
			if ( $page === 'is_post_page' ) {
				wp_nonce_field( 'inpost_seo_save', 'hmpl_ad_inpost_seo_nonce' );
			} else {
				// This shouldn't happen. TODO: Add warning notice?
				return '';
			}
		} else {
			if ( isset( $_GET['taxonomy'] ) ) {
				if ( $_GET['taxonomy'] === 'category' ) {
					$type = __( 'Category', 'autodescription' );
				} else if ( $_GET['taxonomy'] === 'post_tag' ) {
					$type = __( 'Tag', 'autodescription' );
				} else {
					$type = __( 'Theme', 'autodescription' );
				}

				?><h3><?php printf( __( '%s SEO Settings', 'autodescription' ), $type );  ?></h3><?php

				$nobox = true;
			} else {
				return ''; // This shouldn't happen.
			}
		}

		//* Get the language the Google page should assume.
		$language = $this->google_language();

		if ( $nobox ) {
			//* START Taxonomies
			$ad_doctitle = isset( $object->admeta['doctitle'] ) ? $object->admeta['doctitle'] : '';
			$ad_description = isset( $object->admeta['description'] ) ? $object->admeta['description'] : '';
			$ad_noindex = isset( $object->admeta['noindex'] ) ? $object->admeta['noindex'] : '';
			$ad_nofollow = isset( $object->admeta['nofollow'] ) ? $object->admeta['nofollow'] : '';
			$ad_noarchive = isset( $object->admeta['noarchive'] ) ? $object->admeta['noarchive'] : '';
			$ad_savedflag = isset( $object->admeta['saved_flag'] ) ? $object->admeta['saved_flag'] : false;
			$flag = $ad_savedflag ? true : false;

			//* Genesis data fetch
			$genesis = $this->is_theme( 'genesis' );
			if ( $genesis && !$flag ) {
				$ad_doctitle 	= ! empty ( $ad_doctitle ) 		? $ad_doctitle		: $object->meta['doctitle'];
				$ad_description = ! empty ( $ad_description )	? $ad_description	: $object->meta['description'];
				$ad_noindex 	= ! empty ( $ad_noindex )		? $ad_noindex		: $object->meta['noindex'];
				$ad_nofollow 	= ! empty ( $ad_nofollow )		? $ad_nofollow		: $object->meta['nofollow'];
				$ad_noarchive 	= ! empty ( $ad_noarchive )		? $ad_noarchive		: $object->meta['noarchive'];
			}

			$term_id = $object->term_id;
			$taxonomy = $object->taxonomy;

			// $generated_doctitle = $this->title(''); // will return empty here
			$generated_doctitle = $this->title( '', '', '', $term_id, $taxonomy );
			$generated_description = $this->generate_description( '', $term_id, $taxonomy );

			/**
			 * Calculate true Title length
			 *
			 * @since 2.2.4
			 */
			$blog_name = get_bloginfo( 'name', 'display' );
			$title_seperator = $this->title_seperator;
			$sep_option = $this->get_option( 'title_seperator' );
			$sep = array_search( $sep_option, array_flip( $title_seperator ), false );
			$tit_len_additions = ' ' . $sep . ' ' . $blog_name;

			$ad_doctit_parsed_len = ! empty( $ad_doctitle ) ? $ad_doctitle . $tit_len_additions : '';

			$ad_doctitle_len	= ! empty ( $ad_doctitle ) 		? $ad_doctit_parsed_len	: $generated_doctitle;
			$ad_description_len = ! empty ( $ad_description )	? $ad_description		: $generated_description;

			/**
			 * Generate static placeholder for when title is emptied
			 *
			 * @since 2.2.4
			 */
			$doctitle_placeholder = $generated_doctitle;
			$description_placeholder = $generated_description;

			?>
			<table class="form-table">
				<tbody>
					<tr class="form-field">
						<th scope="row" valign="top">
							<label for="autodescription-meta[doctitle]">
								<strong><?php printf( __( '%s Title', 'autodescription' ), $type ); ?></strong>
								<a href="https://support.google.com/webmasters/answer/35624?hl=<?php echo $language; ?>#3" target="_blank" title="<?php _e( 'Recommended Length: 50 to 55 characters', 'autodescription' ) ?>">[?]</a>
							</label>
						</th>
						<td>
							<input name="autodescription-meta[doctitle]" id="autodescription-meta[doctitle]" type="text" placeholder="<?php echo $doctitle_placeholder ?>" value="<?php echo esc_attr( $ad_doctitle ); ?>" size="40" />
							<p class="description"><?php printf( __( 'Characters Used: %s', 'autodescription' ), '<span id="autodescription-meta[doctitle]_chars">'. mb_strlen( $ad_doctitle_len ) .'</span>' ); ?></p>
						</td>
					</tr>

					<tr class="form-field">
						<th scope="row" valign="top">
							<label for="autodescription-meta[description]">
								<strong><?php printf( __( '%s Meta Description', 'autodescription' ), $type ); ?></strong>
								<a href="https://support.google.com/webmasters/answer/35624?hl=<?php echo $language; ?>#1" target="_blank" title="<?php _e( 'Recommended Length: 145 to 155 characters', 'autodescription' ) ?>">[?]</a>
							</label>
						</th>
						<td>
							<textarea name="autodescription-meta[description]" id="autodescription-meta[description]" placeholder="<?php echo $description_placeholder ?>" rows="5" cols="50" class="large-text"><?php echo esc_html( $ad_description ); ?></textarea>
							<p class="description"><?php printf( __( 'Characters Used: %s', 'autodescription' ), '<span id="autodescription-meta[description]_chars">'. mb_strlen( $ad_description_len ) .'</span>' ); ?></p>
						</td>
					</tr>

					<tr>
						<th scope="row" valign="top"><?php _e( 'Robots Meta Settings', 'autodescription' ); ?></th>
						<td>
							<label for="autodescription-meta[noindex]"><input name="autodescription-meta[noindex]" id="autodescription-meta[noindex]" type="checkbox" value="1" <?php checked( $ad_noindex ); ?> />
								<?php printf( __( 'Apply %s to this %s', 'autodescription' ), $this->code_wrap( 'noindex' ), $type ); ?>
								<a href="https://support.google.com/webmasters/answer/93710?hl=<?php echo $language; ?>" target="_blank" title="<?php printf( __( 'Tell Search Engines not to show this page in their search results', 'autodescription' ) ) ?>">[?]</a>
							</label>

							<br />

							<label for="autodescription-meta[nofollow]"><input name="autodescription-meta[nofollow]" id="autodescription-meta[nofollow]" type="checkbox" value="1" <?php checked( $ad_nofollow ); ?> />
								<?php printf( __( 'Apply %s to this %s', 'autodescription' ), $this->code_wrap( 'nofollow' ), $type ); ?>
								<a href="https://support.google.com/webmasters/answer/96569?hl=<?php echo $language; ?>" target="_blank" title="<?php printf( __( 'Tell Search Engines not to follow links on this page', 'autodescription' ) ) ?>">[?]</a>
							</label>

							<br />

							<label for="autodescription-meta[noarchive]"><input name="autodescription-meta[noarchive]" id="autodescription-meta[noarchive]" type="checkbox" value="1" <?php checked( $ad_noarchive ); ?> />
								<?php printf( __( 'Apply %s to this %s', 'autodescription' ), $this->code_wrap( 'noarchive' ), $type ); ?>
								<a href="https://support.google.com/webmasters/answer/79812?hl=<?php echo $language; ?>" target="_blank" title="<?php printf( __( 'Tell Search Engines not to save a cached copy this page', 'autodescription' ) ) ?>">[?]</a>
							</label>

							<?php // Saved flag, if set then it won't fetch for genesis meta anymore ?>
							<label class="hidden" for="autodescription-meta[saved_flag]">
								<input name="autodescription-meta[saved_flag]" id="autodescription-meta[saved_flag]" type="checkbox" value="1" checked='checked' />
							</label>
						</td>
					</tr>
				</tbody>
			</table>
		<?php
		} else {
			//* START Posts/Pages
		//	$post_id = get_the_ID();
			/**
			 * Now uses get_queried_object_id()
			 * @since 2.2.8
			 */
			$post_id = get_queried_object_id() ? get_queried_object_id() : get_the_ID();
			$title = $this->get_custom_field( '_genesis_title' );

			$doctitle_placeholder_use = false;

			$doctitle_placeholder = '';

			$description = $this->get_custom_field( '_genesis_description' );

			/**
			 * Generate static title placeholder for when title is emptied
			 *
			 * @since 2.2.4
			 *
			 * Fetch description from Home Page SEO Settings placeholder if it exists.
			 * @since 2.2.5
			 *
			 * Generate description for Posts Page if selected in customizer.
			 * @since 2.2.8
			 */
			if ( $post_id == get_option( 'page_on_front' ) ) {
				$doctitle_placeholder = $this->title( '', '', '', '', '', true, true );

				$description_placeholder = ! empty ( $description ) ? '' : $this->generate_description( '', $post_id );
			} else if ( $post_id == get_option( 'page_for_posts' ) && 'page' == get_option( 'show_on_front' ) ) {
				$doctitle_placeholder = $this->title( '', '', '', '', '', '', true );

				$description_placeholder = ! empty ( $description ) ? '' : $this->generate_description( '', $post_id );
			} else {
				$doctitle_placeholder = $this->title( '', '', '', '', '', '', true );
				$description_placeholder = ! empty ( $description ) ? '' : $this->generate_description();
			}

			if ( empty ( $title ) ) {
				//* We're going to use the placeholder now for title length calculation.
				$doctitle_placeholder_use = true;
			}

			$canonical =  $this->get_custom_field( '_genesis_canonical_uri' );
			$canonical_placeholder = ! empty ( $canonical ) ? '' : $this->the_url('', $post_id );

			/**
			 * Calculate true Title length
			 *
			 * @since 2.2.4
			 */
			$blog_name = get_bloginfo( 'name', 'display' );
			$title_seperator = $this->title_seperator;
			$sep_option = $this->get_option( 'title_seperator' );
			$sep = array_search( $sep_option, array_flip( $title_seperator ), false );
			$tit_len_additions = ' ' . $sep . ' ' . $blog_name;

			$doctit_parsed_len = ! empty( $title ) ? $title . $tit_len_additions : '';

			$tit_len = $doctitle_placeholder_use ? $doctitle_placeholder : $doctit_parsed_len;
			$desc_len = ! empty( $description_placeholder ) ? $description_placeholder : $this->get_custom_field( '_genesis_description' );

			?>
			<p>
				<label for="autodescription_title"><strong><?php printf( __( 'Custom %s Title', 'autodescription' ), $type ); ?></strong>
					<a href="https://support.google.com/webmasters/answer/35624?hl=<?php echo $language; ?>#3" target="_blank" title="<?php _e( 'Recommended Length: 50 to 55 characters', 'autodescription' ) ?>">[?]</a>
					<span class="description"><?php printf( __( 'Characters Used: %s', 'autodescription' ), '<span id="autodescription_title_chars">'. mb_strlen( $tit_len ) .'</span>' ); ?></span>
				</label>
			</p>
			<p>
				<input class="large-text" type="text" name="autodescription[_genesis_title]" id="autodescription_title" placeholder="<?php echo $doctitle_placeholder ?>" value="<?php echo esc_attr( $this->get_custom_field( '_genesis_title' ) ); ?>" />
			</p>

			<p>
				<label for="autodescription_description">
					<strong><?php printf( __( 'Custom %s Description', 'autodescription' ), $type ); ?></strong>
					<a href="https://support.google.com/webmasters/answer/35624?hl=<?php echo $language; ?>#1" target="_blank" title="<?php _e( 'Recommended Length: 145 to 155 characters', 'autodescription' ) ?>">[?]</a>
					<span class="description"><?php printf( __( 'Characters Used: %s', 'autodescription' ), '<span id="autodescription_description_chars">'. mb_strlen( $desc_len ) .'</span>' ); ?></span>
				</label>
			</p>
			<p>
				<textarea class="large-text" name="autodescription[_genesis_description]" id="autodescription_description" placeholder="<?php echo $description_placeholder ?>" rows="4" cols="4"><?php echo esc_textarea( $this->get_custom_field( '_genesis_description' ) ); ?></textarea>
			</p>

			<p>
				<label for="autodescription_canonical">
					<strong><?php _e( 'Custom Canonical URL', 'autodescription' ); ?></strong>
					<a href="https://support.google.com/webmasters/answer/139066?hl=<?php echo $language; ?>" target="_blank" title="&lt;link rel=&quot;canonical&quot; /&gt;">[?]</a>
				</label>
			</p>
			<p>
				<input class="large-text" type="text" name="autodescription[_genesis_canonical_uri]" id="autodescription_canonical" placeholder="<?php echo $canonical_placeholder ?>" value="<?php echo esc_url( $this->get_custom_field( '_genesis_canonical_uri' ) ); ?>" />
			</p>

			<p><strong><?php _e( 'Robots Meta Settings', 'autodescription' ); ?></strong></p>
			<p>
				<label for="autodescription_noindex"><input type="checkbox" name="autodescription[_genesis_noindex]" id="autodescription_noindex" value="1" <?php checked( $this->get_custom_field( '_genesis_noindex' ) ); ?> />
					<?php printf( __( 'Apply %s to this %s', 'autodescription' ), $this->code_wrap( 'noindex' ), $type ); ?>
					<a href="https://support.google.com/webmasters/answer/93710?hl=<?php echo $language; ?>" target="_blank" title="<?php printf( __( 'Tell Search Engines not to show this page in their search results', 'autodescription' ) ) ?>">[?]</a>
				</label>

				<br />

				<label for="autodescription_nofollow"><input type="checkbox" name="autodescription[_genesis_nofollow]" id="autodescription_nofollow" value="1" <?php checked( $this->get_custom_field( '_genesis_nofollow' ) ); ?> />
					<?php printf( __( 'Apply %s to this %s', 'autodescription' ), $this->code_wrap( 'nofollow' ), $type ); ?>
					<a href="https://support.google.com/webmasters/answer/96569?hl=<?php echo $language; ?>" target="_blank" title="<?php printf( __( 'Tell Search Engines not to follow links on this page', 'autodescription' ) ) ?>">[?]</a>
				</label>

				<br />

				<label for="autodescription_noarchive"><input type="checkbox" name="autodescription[_genesis_noarchive]" id="autodescription_noarchive" value="1" <?php checked( $this->get_custom_field( '_genesis_noarchive' ) ); ?> />
					<?php printf( __( 'Apply %s to this %s', 'autodescription' ), $this->code_wrap( 'noarchive' ), $type ); ?>
					<a href="https://support.google.com/webmasters/answer/79812?hl=<?php echo $language; ?>" target="_blank" title="<?php printf( __( 'Tell Search Engines not to save a cached copy this page', 'autodescription' ) ) ?>">[?]</a>
				</label>

				<?php // Saved flag, if set then it won't fetch for genesis meta anymore ?>
				<label class="hidden" for="autodescription_saved_flag">
					<input name="autodescription[saved_flag]" id="autodescription[saved_flag]" type="checkbox" value="1" checked='checked' />
				</label>
			</p>

			<p><strong><?php _e( 'Local Search Settings', 'autodescription' ); ?></strong></p>
			<p>
				<label for="autodescription_exclude_local_search"><input type="checkbox" name="autodescription[exclude_local_search]" id="autodescription_exclude_local_search" value="1" <?php checked( $this->get_custom_field( 'exclude_local_search' ) ); ?> />
					<?php printf( __( 'Exclude this %s from local search', 'autodescription' ), $type ); ?>
					<span title="<?php printf( __( 'This excludes this %s from local on-site search results.', 'autodescription' ), $type ) ?>">[?]</a>
				</label>
			</p>

			<p>
				<label for="autodescription_redirect">
					<strong><?php _e( 'Custom 301 Redirect URL', 'autodescription' ); ?></strong>
					<a href="https://support.google.com/webmasters/answer/93633?hl=<?php echo $language; ?>" target="_blank" title="301 Redirect">[?]</a>
				</label>
			</p>
			<p>
				<input class="large-text" type="text" name="autodescription[redirect]" id="genesis_redirect" value="<?php echo esc_url( $this->get_custom_field( 'redirect' ) ); ?>" />
			</p>
		<?php
		}
	}

}
