<?php
/**
 * Class AutoDescription_Init
 *
 * Initializes the plugin
 * Outputs all data in front-end header
 *
 * @since 2.1.6
 */
class AutoDescription_Init {

	/**
	 * Constructor. Init actions.
	 *
	 * @since 2.1.6
	 */
	public function __construct() {

		add_action( 'init', array( $this, 'autodescription_run' ), 1 );
		add_action( 'template_redirect', array( $this, 'custom_field_redirect') );

	}

	/**
	 * Run the plugin
	 *
	 * @since 1.0.0
	 */
	public function autodescription_run() {

		/**
		 * Remove Genesis output
		 *
		 * Removed.
		 * @since 2.2.4
		 */
		/*
		$genesis = $this->is_theme( 'genesis' );

		if ( $genesis ) {
			remove_action( 'genesis_meta', 'genesis_seo_meta_description', 10 ); //genesis seo
			remove_action( 'genesis_meta','genesis_seo_meta_keywords' ); //clean up residue (meta tags)
			remove_action( 'wp_head','genesis_canonical', 5 ); //genesis canonical
			remove_action( 'genesis_meta', 'genesis_robots_meta' ); //genesis robots
		}
		*/

		//* Remove canonical header tag from WP
		remove_action( 'wp_head', 'rel_canonical' );

		//* Remove generator tag from WP
		add_filter( 'the_generator', '__return_false' );

		//* Override WordPress Title
		add_filter( 'wp_title', array( $this, 'title_from_cache' ), 9, 3 );

		/**
		 * Override Woo Themes Title
		 */
		add_filter( 'woo_title', array( $this, 'title_from_cache'), 99 );

		/**
		 * Don't do anything on preview
		 * @since 2.2.4
		 */
		if ( !is_preview() ) {
			$genesis = $this->is_theme( 'genesis' );

			if ( $genesis ) {
				add_action( 'genesis_meta', array( $this, 'html_output' ), 5 );
			} else {
				add_action( 'wp_head', array( $this, 'html_output' ), -1 );
			}
		}
	}

	/**
	 * Header actions.
	 *
	 * @uses The_SEO_Framework_Load::call_function()
	 *
	 * @param string|array $args the arguments that will be passed
	 * @param bool $before if the header actions should be before or after the SEO Frameworks output
	 *
	 * @since 2.2.6
	 *
	 * @return string|null
	 */
	public function header_actions( $args = '', $before = true ) {

		$output = '';

		//* Placeholder callback and args.
		$functions = array(
						'callback'	=> '',
						'args'		=> '',
					);

		//* Marked for removal @since 2.3.5
		$tag_old = $before ? 'seo_framework_before_output' : 'seo_framework_after_output';
		//* @since 2.3.0 filter.
		$tag = $before ? 'the_seo_framework_before_output' : 'the_seo_framework_after_output';

		//* Marked for removal @since 2.3.5
		$filter = apply_filters( $tag_old, $functions );
		//* @since 2.3.0 filter.
		$filter = (array) apply_filters( $tag, $filter );

		$functions = wp_parse_args( $args, $filter );

		if ( is_array( $functions ) ) {
			foreach ( $functions as $function ) {
				$arguments = isset( $function['args'] ) ? $function['args'] : '';

				if ( isset( $function['callback'] ) )
					$output .= $this->call_function( $function['callback'], '2.2.6', $arguments );

			}
		}

		return $output;
	}

	/**
	 * Output the header meta and script
	 *
	 * @since 1.0.0
	 *
	 * @param blog_id : the blog id
	 *
	 * @filter hmpl_ad_pre 	: Adds content before
	 * 						: @param before
	 *						: cached
	 * @filter hmpl_ad_pro 	: Adds content after
	 *						: @param after
	 *						: cached
	 *
	 * @uses hmpl_ad_description()
	 * @uses $this->og_image()
	 * @uses $this->og_locale()
	 * @uses $this->og_type()
	 * @uses $this->og_title()
	 * @uses $this->og_description()
	 * @uses $this->og_url()
	 * @uses $this->og_sitename()
	 * @uses $this->ld_json()
	 * @uses $this->canonical()
	 *
	 * Echos output.
	 */
	public function html_output() {
		global $blog_id,$paged,$page;

		$page_id = get_queried_object_id();

		/**
		 * Give each paged pages/archives a different cache key.
		 * @since 2.2.6
		 */
		$page = isset( $page ) ? (string) $page : '0';
		$paged = isset( $paged ) ? (string) $paged : '0';

		$output = wp_cache_get( 'seo_framework_output_' . $blog_id . '_' . $page_id . '_' . $paged . '_' . $page, 'theseoframework' );
		if ( false === $output ) {

			//* Marked for removal @since 2.3.5
			$indicator = apply_filters( 'hmpl_ad_indicator', true );
			//* @since 2.3.0 filter.
			$indicator = (bool) apply_filters( 'the_seo_framework_indicator', $indicator );

			$indicatorbefore = '';
			$indicatorafter = '';

			if ( $indicator !== false ) {
				$indicatorbefore = '<!-- Start The Seo Framework by Sybre Waaijer -->' . "\r\n";
				$indicatorafter = '<!-- End The Seo Framework by Sybre Waaijer -->' . "\r\n";
			}

			//* Marked for removal @since 2.3.5
			$before = apply_filters( 'hmpl_ad_pre', '' );
			//* @since 2.3.0 filter.
			$before = (string) apply_filters( 'the_seo_framework_pre', $before );

			$before_actions = $this->header_actions( '', true );

			$robots = $this->robots();

			//* Limit processing on 404 or search
			if ( !is_404() && !is_search() ) {
				$output	= $this->the_description()
						. $this->og_image()
						. $this->og_locale()
						. $this->og_type()
						. $this->og_title()
						. $this->og_description()
						. $this->og_url()
						. $this->og_sitename()
						. $this->facebook_publisher()
						. $this->facebook_author()
						. $this->facebook_app_id()
						. $this->article_published_time()
						. $this->article_modified_time()
						. $this->twitter_card()
						. $this->twitter_site()
						. $this->twitter_creator()
						. $this->twitter_title()
						. $this->twitter_description()
						. $this->twitter_image()
						. $this->shortlink()
						. $this->canonical()
						. $this->paged_urls()
						. $this->ld_json()
						. $this->google_site_output()
						. $this->bing_site_output()
						;
			} else {
				$output	= $this->og_locale()
						. $this->og_type()
						. $this->og_title()
						. $this->og_url()
						. $this->og_sitename()
						. $this->canonical()
						. $this->ld_json()
						. $this->google_site_output()
						. $this->bing_site_output()
						;
			}

			$after_actions = $this->header_actions( '', false );

			//* Marked for removal @since 2.3.5
			$after = apply_filters( 'hmpl_ad_pro', '' );
			//* @since 2.3.0 filter.
			$after = (string) apply_filters( 'the_seo_framework_pro', $after );

			/**
			 * This should get its own function?
			 * Or its own plugin. =)
			 *
			 * @see https://wordpress.org/plugins/generator-the-seo-framework/
			 */
			//* Marked for removal @since 2.3.5
			$generator = apply_filters( 'hmpl_ad_generator', $generator = '' );
			//* @since 2.3.0 filter.
			$generator = (string) apply_filters( 'the_seo_framework_generator_tag', $generator );

			if ( !empty( $generator ) )
				$generator = '<meta name="generator" content="' . esc_attr( $generator ) . '" />' . "\r\n";

			$output = "\r\n" . $indicatorbefore . $robots . $before . $before_actions . $output . $after_actions . $after . $generator . $indicatorafter . "\r\n";

			wp_cache_set( 'seo_framework_output_' . $blog_id . '_' . $page_id . '_' . $paged . '_' . $page, $output, 'theseoframework', 86400 );
		}

		echo $output;
	}

	/**
	 * Redirect singular page to an alternate URL.
	 *
	 * Called outside autodescription_run
	 *
	 * @since 2.0.9
	 */
	public function custom_field_redirect() {

		//* Prevent redirect from options on uneditable pages or when this plugin is set to be disabled
		if ( ! is_singular() )
			return;

		$url = $this->get_custom_field( 'redirect' );

		if ( $url ) {

			//* Marked for removal @since 2.3.5
			$allow_external = apply_filters( 'hmpl_ad_301_external', true );
			//* @since 2.3.0 filter.
			$allow_external = (bool) apply_filters( 'the_seo_framework_allow_external_redirect', $allow_external );

			/**
			 * If the URL is made relative, prevent scheme issues
			 * Always do this if IS_HMPL
			 *
			 * Removes http:// and https://
			 *
			 * esc_url_raw uses is_ssl() to make the url good again :)
			 *
			 * @todo consider this to be an "always on" feature, only if link is pointing to the current domain of course.
			 */
			if ( ! $allow_external || ( defined( 'IS_HMPL' ) && IS_HMPL ) ) {
				$pattern 	= 	'/'
							.	'(((http)(s)?)\:)' 	// 1: http: https:
							. 	'(\/\/)'			// 2: slash slash
							.	'/s'
							;

				$url = preg_replace( $pattern, '', $url );
			}

			wp_redirect( esc_url_raw( $url ), 301 );
			exit;

		}

	}

}
