<?php
/**
 * Class AutoDescription_Generate
 *
 * Generates SEO data based on content
 * Returns strings/arrays
 *
 * @since 2.1.6
 */
class AutoDescription_Generate extends AutoDescription_PostData {

	/**
	 * Constructor, load parent constructor
	 */
	public function __construct() {
		parent::__construct();
	}

	/**
	 * Create description
	 *
	 * @since 1.0.0
	 * @param string $description the description.
	 * @param int $id the term or page id.
	 * @param string $taxonomy taxonomy name.
	 * @param bool $is_home We're generating for the home page.
	 *
	 * @todo 	Save output into the option for reduced load.
	 * 			Counter intuitive?
	 *
	 * @return string output	The description, might be unsafe for html output
	 */
	 public function generate_description( $description = '', $id = '', $taxonomy = '', $is_home = '' ) {
		global $wp_query;

		$homeisblog = false;
		$page_for_posts = false;

		if ( empty( $description ) ) {
			//* Fetch from options, if any.

			$genesis = $this->is_theme( 'genesis' );

			if ( is_front_page() || ( ! empty( $id ) && empty( $taxonomy ) && $id == get_option( 'page_on_front' ) ) ) {
				$description = $this->get_option( 'homepage_description' ) ? $this->get_option( 'homepage_description' ) : $description;

				/**
				 * Removed
				 *
				 * @since 2.2.8
				 */
				/* $flag = $this->get_custom_field( 'saved_flag' ) ? true : false;

				//* Genesis fallback until option is created
				//	if ( empty($description) )
				// *
				if ( $genesis && empty( $description ) && !$flag )
					$description = genesis_get_seo_option( 'home_description' ) ? genesis_get_seo_option( 'home_description' ) : '';
				*/
			}

			if ( is_singular() && empty( $description ) ) {
				//* Bugfix 2.2.7 run only if description is stil empty from home page.
				$description = $this->get_custom_field( '_genesis_description' ) ? $this->get_custom_field( '_genesis_description' ) : $description;

				$flag = $this->get_custom_field( 'saved_flag' ) ? true : false;

				//* Genesis fallback
				if ( $genesis && empty( $description ) && !$flag )
					$description = genesis_get_custom_field( '_genesis_description' );
			}

			if ( is_category() ) {
				$term = $wp_query->get_queried_object();

				$description = ! empty( $term->admeta['description'] ) ? $term->admeta['description'] : $description;

				$flag = $term->admeta['saved_flag'] ? true : false;

				if ( empty( $description ) && $genesis && !$flag )
					$description = ! empty( $term->meta['description'] ) ? $term->meta['description'] : $description;
			}

			if ( is_tag() ) {
				$term = $wp_query->get_queried_object();

				$description = ! empty( $term->admeta['description'] ) ? $term->admeta['description'] : $description;

				$flag = $term->admeta['saved_flag'] ? true : false;

				if ( empty( $description ) && $genesis && !$flag )
					$description = ! empty( $term->meta['description'] ) ? $term->meta['description'] : $description;
			}

			if ( is_tax() ) {
				$term = get_term_by( 'slug', get_query_var( 'term' ), get_query_var( 'taxonomy' ) );
				$description = ! empty( $term->admeta['description'] ) ? wp_kses_stripslashes( wp_kses_decode_entities( $term->admeta['description'] ) ) : $description;

				$flag = $term->admeta['saved_flag'] ? true : false;

				if ( empty( $description ) && $genesis && !$flag )
					$description = ! empty( $term->meta['description'] ) ? $term->meta['description'] : $description;
			}

			if ( is_author() ) {
				$user_description = get_the_author_meta( 'meta_description', (int) get_query_var( 'author' ) );

				$description = $user_description ? $user_description : $description;
			}

			/**
			 * Fetch description from blog page.
			 * Tell the autodescription we're on a blogpage now.
			 *
			 * @bug in WordPress? page_for_posts returns an integer even if it's not assigned.
			 * @fix: parse get_option( 'show_on_front' )
			 *
			 * @since 2.2.8
			 */
			if ( !is_front_page() && empty( $description ) && ( ( get_the_ID() == get_option( 'page_for_posts' ) ) || ( ! empty( $id ) && $id == get_option( 'page_for_posts' ) ) ) && 'page' == get_option( 'show_on_front' ) ) {
				$description = $this->get_custom_field( '_genesis_description', get_queried_object_id() ) ? $this->get_custom_field( '_genesis_description', get_queried_object_id() ) : $description;
				$page_for_posts = true;
			}

			/**
			 * Removed
			 *
			 * @since 2.2.8
			 */
			/*
			if ( $genesis ) {
				if ( is_post_type_archive() && genesis_has_post_type_archive_support() ) {
					//* Genesis fallback until option is created
					if ( $genesis && empty( $description ) && !$flag )
						$description = genesis_get_cpt_option( 'description' ) ? genesis_get_cpt_option( 'description' ) : $description; // genesis only, for now
				}
			}
			*/

		}

		//* Still no description found? Create an auto description based on content.
		if ( empty( $description ) || !is_string( $description ) ) {
		//	global $blog_id,$post;
			global $blog_id;

			/**
			 * The Page ID is only used for caching, or determining the home page. Nothing else.
			 *
			 * The term and post objects are however used for generation.
			 *
			 * Now uses get_queried_object_id()
			 * @since 2.2.8
			 */
			$page_id = get_queried_object_id() ? get_queried_object_id() : get_the_ID();

			if ( ! empty( $id ) && !empty( $taxonomy ) ) {
				//* If on taxonomy
				$term = get_term_by( 'id', $id, $taxonomy, OBJECT );
				$page_id = $term ? $taxonomy . '_' . $id : $page_id;
			} else if ( $is_home && !empty( $id ) && 'page' == get_option( 'show_on_front' ) ) {
				//* If on front page as page
				$page_id = $id;
				$post = get_post( $page_id, OBJECT );
			}

			/**
			 * We're on the home page blog now. So let's create something special.
			 * @since 2.2.5
			 *
			 * Check for is_front_page()
			 * @since 2.2.6
			 *
			 * Seperate from if else loop
			 * @since 2.2.8
			 */
			if ( $page_id === false || is_front_page() || $page_id == get_option( 'page_on_front' ) ) {
				$homeisblog = true;

				/**
				 * Removed because we use queried object ID now.
				 * Also prevents conflicts.
				 *
				 * @since 2.2.8
				 */
				//	$page_id = -1;
			}

			/**
			 * Cache the generated description.
			 */
			$description = wp_cache_get( 'auto_description_' . $blog_id . '_' . $page_id, 'autodescription' );
			if ( false === $description ) {

				if ( ! $homeisblog ) {
					if ( !empty( $term ) && is_object( $term ) ) {
						$title = !empty( $term->name ) ? $term->name : $term->slug;
						$excerpt = !empty( $term->description ) ? $term->description : $this->get_excerpt_by_id(); // The latter will return empty in admin pages
					} else if ( $page_for_posts ) {
						//* @since 2.2.8
						$title = $this->get_custom_field( '_genesis_title', $page_id ) ? $this->get_custom_field( '_genesis_title', $page_id ) : get_the_title( $page_id );

						// @TODO create option.
						$title = __( 'Latest posts:', 'autodescription' ) . ' ' . $title;
						$excerpt = '';
					} else {
						$title = $this->get_custom_field( '_genesis_title', $page_id ) ? $this->get_custom_field( '_genesis_title', $page_id ) : get_the_title( $page_id );
						$excerpt = $this->get_excerpt_by_id( '', $page_id );
					}
				} else {
					$title = get_bloginfo( 'description', 'raw' );
					$excerpt = '';
				}

				/**
				 * Use Untitled on empty titles.
				 * @since 2.2.8
				 */
				$title = empty( $title ) ? __( 'Untitled', 'autodescription' ) : $title;

				$blogname = get_bloginfo( 'name', 'raw' );
				$excut = '';

				// @TODO create option for this
				$sep = '|';
				$on = _x( 'on', 'Placement. e.g. Post Title "on" Blog Name', 'autodescription' );

				//* Get max char length
				//* 152 whilst account for the added ... makes 155
				$max_char_length = 152 - mb_strlen( $title . $on . $blogname );
				$excerptlength = mb_strlen( $excerpt );

				// Trunculate if the excerpt is longer than the max char length
				if ( $excerptlength > $max_char_length ) {

					// Cut string to fit $max_char_length
					$subex = mb_substr( $excerpt, 0, $max_char_length );
					// Split words in array
					$exwords = explode( ' ', $subex );
					// Calculate if last word exceeds.
					$excut = - ( mb_strlen( $exwords[ count( $exwords ) - 1 ] ) );

					if ( $excut < 0 ) {
						//* Cut out exceeding word.
						$excerpt = mb_substr( $subex, 0, $excut );
					} else {
						// We're all good here, continue.
						$excerpt = $subex;
					}
					// Replace exceeding word with ...
					$exceed = '...';

					/**
					 * Moved up, to reduce processing power
					 *
					 * @since 2.2.8
					 */
					$excerpt = str_replace( ' ...', '...', $excerpt . $exceed );
				}

				if ( !empty( $excerpt ) ) {
					$description = sprintf( '%s %s %s %s %s', $title, $on, $blogname, $sep, $excerpt );
				} else {
					$description = sprintf( '%s %s %s', $title, $on, $blogname );
				}

				wp_cache_set( 'auto_description_' . $blog_id . '_' . $page_id, $description, 'autodescription' );
			}

		}

		/**
		 * @uses esc_attr.
		 * @since 2.3.0
		 */
		$description = esc_attr( trim( $description ) );

		return $description;
	}

	/**
	 * Get the title
	 *
	 * @TODO unspaghettify a.k.a create functions from if statements.
	 * @TODO reduce params?
	 *
	 * @since 1.0.0
	 *
	 * @param string $title The Title to return
	 * @param string $sep The Title sepeartor
	 * @param string $seplocation The Title sepeartor location ( accepts 'left' or 'right' )
	 * @param int $term_id The Taxonomy Term ID
	 * @param string $taxonomy The Taxonomy name
	 * @param bool $page_on_front Page on front condition for example generation
	 * @param bool $placeholder Generate placeholder, ignoring options.
	 * @param bool $notagline Generate title without tagline.
	 * @param bool $meta Continue normally if set to true, regardless of theme doing it right. Used in og:title/twitter:title
	 * @param bool $term_doingitright Fetch the title as Category or Term for The SEO Bar.
	 *
	 * @return string $title Title
	 */
	public function title( $title = '', $sep = '', $seplocation = '', $term_id = '', $taxonomy = '', $page_on_front = false, $placeholder = false, $notagline = false, $meta = false ) {

		if ( is_feed() )
			return trim( $title );

		/**
		 * Set to true if wp_title is used incorrectly.
		 *
		 * @param bool $doingitwrong
		 *
		 * @since 2.2.5
		 */
		$doingitwrong = false;

		/**
		 * Remove seperator if true.
		 *
		 * @param bool $no_sep
		 *
		 * @since 2.2.5
		 */
		$sep_replace = false;

		/**
		 * Add doing it wrong notice for better SEO consistency.
		 *
		 * Only when in wp_title.
		 *
		 * @since 2.2.5
		 */
		if ( ! empty( $seplocation ) && doing_filter( 'wp_title' ) && ! $this->detect_theme_support( 'title-tag' ) && ! $meta ) {
			// Don't disturb the precious title when WP_DEBUG is on.
			add_action( 'wp_footer', array( $this, 'title_doing_it_wrong' ), 20 );

			//* Tell the rest of this function it's been done wrong.
			$doingitwrong = true;

			//* Maybe remove seperator.
			if ( ( !empty ( $sep ) ) ) {
				$sep_replace = true;
				$sep_to_replace = $sep;
			}
		}

		/**
		 * Detect if placeholder is being generated.
		 *
		 * @since 2.2.4
		 *
		 * Moved up for early return
		 * @since 2.2.5
		 */
		if ( empty( $title ) && $placeholder && !empty( $term_id ) ) {
			$title = get_the_title( $term_id );
		} else if ( empty( $title ) && $placeholder && $page_on_front ) {
			$title = get_the_title( get_option( 'page_on_front' ) );
		} else if ( empty( $title ) && $placeholder ) {
			global $post;
			$title = get_the_title( $post );
		}

		// End early if the request is the Title only
		if ( $notagline )
			return esc_html( trim( $title ) );

		/**
		 * Overwrite title here, prevents duplicate title issues, since we're working with a filter.
		 *
		 * @since 2.2.2
		 *
		 * Only do so when the title is OK.
		 * @since 2.2.5
		 */
		if ( ! $doingitwrong )
			$title = '';

		/**
		 * Tagline conditional for homepage
		 *
		 * @since 2.2.2
		 */
		$add_tagline = 0;

		/**
		 * Tagline conditional extension for homepage
		 *
		 * @since 2.2.2
		 */
		$get_custom_field = true;

		$blogname = get_bloginfo( 'name', 'raw' );

		/**
		 * Cache the seplocation for is_home()
		 *
		 * @since 2.2.2
		 */
		$seplocation_home = $seplocation;

		/**
		 * Filters the seperator
		 *
		 * @since 2.1.8
		 */
		if ( empty ( $seplocation ) || $seplocation !== 'left' || $seplocation !== 'right' ) {
			//* Marked for removal @since 2.3.5
			$seplocation = apply_filters( 'hmpl_ad_title_seplocation', $this->get_option( 'title_location' ) );
			//* @since 2.3.0 filter.
			$seplocation = (string) apply_filters( 'the_seo_framework_title_seplocation', $seplocation );
		}

		$title_seperator = $this->title_seperator;
		$sep_option = $this->get_field_value( 'title_seperator' );
		$sep = array_search( $sep_option, array_flip( $title_seperator ), false );

		/**
		 * Filters the seperator
		 *
		 * @since 2.0.5
		 */
		//* Marked for removal @since 2.3.5
		$sep = apply_filters( 'hmpl_ad_title_seperator', $sep );
		//* @since 2.3.0 filter.
		$sep = (string) apply_filters( 'the_seo_framework_title_seperator', $sep );

		if ( empty( $title ) ) {

			if ( is_front_page() || $page_on_front == true || get_the_ID() == get_option( 'page_on_front' ) ) {
				$add_tagline = $this->get_option( 'homepage_tagline' );

				$tagline = '';

				/**
				 * Add tagline or not based on option
				 *
				 * @since 2.2.2
				 */
				if ( $add_tagline ) {
					$tagline = get_bloginfo( 'description', 'raw' );
				} else {
					$get_custom_field = false;
				}

				$title = ! empty( $tagline ) ? $tagline : '';

				//* Render from function @since 2.2.8
				$home_title = $this->title_for_home();

				$blogname = ! empty( $home_title ) ? $home_title : $blogname;

				if ( empty ( $seplocation_home ) || $seplocation_home !== 'left' || $seplocation_home !== 'right' ) {
					//* Marked for removal @since 2.3.5
					$seplocation = apply_filters( 'hmpl_ad_title_seplocation_front', $this->get_option( 'home_title_location' ) );
					//* @since 2.3.0 filter.
					$seplocation = (string) apply_filters( 'the_seo_framework_title_seplocation_front', $seplocation );
				}
			}

			/**
			 * Combined the statements
			 * @since 2.2.7 && @since 2.2.8
			 */
			if ( is_category() || is_tag() || is_tax() ) {
				$title = $this->title_for_terms( $title );
			}

			/**
			 * Combined the statements
			 * @since 2.2.8
			 */
			if ( is_date() || is_author() ) {
				$title = wp_strip_all_tags( get_the_archive_title() );
			}

			if ( is_404() ) {
				// @TODO create options
				$title = '404';
			}

			if ( is_search() ) {
				// @TODO create options
				$title = __( 'Search results for:', 'autodescription' ) . ' ' . get_search_query();
			}

			// Generate admin placeholder for taxonomies
			if ( !empty( $term_id ) && !empty( $taxonomy ) && empty( $title ) ) {
				$term = get_term_by( 'id', $term_id, $taxonomy, OBJECT );

				if ( !empty( $term ) && is_object( $term ) )
					$term_name = !empty( $term->name ) ? $term->name : $term->slug;

				$tax_type = $term->taxonomy;

				/**
				 * Translations from WP Core function get_the_archive_title()
				 *
				 * @since 2.2.7
				 */
				if ( $tax_type == 'category' ) {
					$title = sprintf( __( 'Category: %s' ), $term_name );
				} elseif ( $tax_type == 'post_tag' ) {
					$title = sprintf( __( 'Tag: %s' ), $term_name );
				} else {
					// Fallback
					$title = __( 'Archives' );
				}
			}

			//* Move this up?
			if ( empty( $title ) && $get_custom_field ) {
				$title = $this->title_from_custom_field();
			}

			//* You forgot to enter a title "anywhere"!
			if ( empty( $title ) && !$sep_replace ) {
				$title = __( 'Untitled', 'autodescription' );
			} elseif ( empty( $title ) && $sep_replace ) {
				//* Include seperator for doing it wrong.
				$title =  __( 'Untitled', 'autodescription' ) . ' ' . $sep_to_replace;
			}

		}

		/**
		 * @since 2.2.5
		 *
		 * Checks the $doingitwrong bool.
		 */
		if ( ! $doingitwrong ) {
			// Doing it right.

			if ( ( is_front_page() || $page_on_front == true ) && ! $add_tagline ) {
				//* Render frontpage output without tagline
				$title = $blogname;
			} else if ( $seplocation == 'right' ) {
				$title = $title . " $sep " . $blogname;
			} else {
				$title = $blogname . " $sep " . $title;
			}

			$title = esc_attr( trim( $title ) );

		} else {
			// Doing it wrong.

			if ( ( is_front_page() || $page_on_front == true ) && ! $add_tagline ) {
				//* Render frontpage output without tagline
				$title = $blogname;
			} else if ( is_front_page() || $page_on_front == true ) {
				//* Remove the title, because most themes think they 'know' how to SEO the front page.
				// Because, most themes know how to make the title 'pretty'. And therefor add all kinds of stuff.
				$title = '';
			} else {
				//* Sep location has no influence.
				if ( ! $sep_replace ) {
					$title = trim( $title ) . " $sep ";
				} else {
					// Add trailing space for the tagline/blogname is stuck onto this part with trim.
					$title = trim( str_replace( $sep_to_replace, $sep, $title ) ) . " ";
				}
			}

			$title = esc_attr( $title );

		}

		return $title;
	}

	/**
	 * Gets the title for the static home page.
	 *
	 * @since 2.2.8
	 *
	 * @return string The Title.
	 */
	public function title_for_home() {
		/**
		 * Get blogname title based on option
		 *
		 * @since 2.2.2
		 */
		$home_title = $this->get_option( 'homepage_title' );

		/**
		 * Fetch from Home Page InPost SEO Box if empty.
		 *
		 * @since 2.2.4
		 *
		 * Add home is page check.
		 * @since 2.2.5
		 */
		if ( empty( $home_title ) && 'page' === get_option( 'show_on_front' ) ) {
			$home_title = $this->get_custom_field( '_genesis_title' ) ? $this->get_custom_field( '_genesis_title' ) : '';

			$genesis = $this->is_theme( 'genesis' );

			if ( empty( $home_title ) && $genesis ) {
				$flag = $this->get_custom_field( 'saved_flag' ) ? true : false;

				if ( !$flag )
					$home_title = genesis_get_custom_field( '_genesis_title' ) ? genesis_get_custom_field( '_genesis_title' ) : $home_title;
			}
		}

		return $home_title;
	}

	/**
	 * Gets the title for Category, Tag or Taxonomy
	 *
	 * @since 2.2.8
	 *
	 * @param string $title the current title.
	 *
	 * @return string The Title.
	 */
	public function title_for_terms( $title ) {

		if ( is_category() || is_tag() ) {
			global $wp_query;

			$term = $wp_query->get_queried_object();

			//* Genesis only check
			$genesis = $this->is_theme( 'genesis' );

			$title = ! empty( $term->admeta['doctitle'] ) ? $term->admeta['doctitle'] : $title;

			$flag = ! empty ( $term->admeta['saved_flag'] ) ? true : false;

			if ( empty( $title ) && $genesis && !$flag )
				$title = ! empty( $term->meta['doctitle'] ) ? $term->meta['doctitle'] : $title;

			if ( empty( $title ) )
				$title = !empty ( $term->name ) ? wp_strip_all_tags( get_the_archive_title() ) : $term->slug;

		} else if ( is_tax() ) {
			$term  = get_term_by( 'slug', get_query_var( 'term' ), get_query_var( 'taxonomy' ) );

			//* Genesis only check
			$genesis = $this->is_theme( 'genesis' );

			$title = ! empty( $term->admeta['doctitle'] ) ? wp_kses_stripslashes( wp_kses_decode_entities( $term->admeta['doctitle'] ) ) : $title;

			$flag = ! empty ( $term->admeta['saved_flag'] ) ? true : false;

			if ( empty( $title ) && $genesis && !$flag )
				$title = ! empty( $term->meta['doctitle'] ) ? wp_kses_stripslashes( wp_kses_decode_entities( $term->meta['doctitle'] ) ) : $title;

			if ( empty( $title ) )
				$title = !empty ( $term->name ) ? wp_strip_all_tags( get_the_archive_title() ) : $term->slug;

		}

		return $title;
	}

	/**
	 * Gets the title from custom field
	 *
	 * @since 2.2.8
	 *
	 * @return string The Title.
	 */
	public function title_from_custom_field() {
		/**
		 * Create something special for blog page.
		 * Only if it's not the home page.
		 *
		 * @since 2.2.8
		 */
		if ( !is_front_page() && get_queried_object_id() == get_option( 'page_for_posts' ) && 'page' == get_option( 'show_on_front' ) ) {
			$title = $this->get_custom_field( '_genesis_title', get_queried_object_id() ) ? $this->get_custom_field( '_genesis_title', get_queried_object_id() ) : get_the_title( get_queried_object_id() );
		} else {
			//* Get title from custom field, empty it if it's not there to override the default title
			$title = $this->get_custom_field( '_genesis_title' ) ? $this->get_custom_field( '_genesis_title' ) : '';
		}

		$genesis = $this->is_theme( 'genesis' );

		if ( empty( $title ) && $genesis ) {
			$flag = $this->get_custom_field( 'saved_flag' ) ? true : false;

			if ( !$flag )
				$title = genesis_get_custom_field( '_genesis_title' ) ? genesis_get_custom_field( '_genesis_title' ) : '';
		}

		if ( empty( $title ) ) {
			global $post;
			$title = get_the_title( $post );
		}

		return $title;
	}

	/**
	 * Creates canonical url
	 *
	 * @uses WPMUdev's domain mapping
	 *
	 * @param string $url the url
	 * @param int $page_id the page id, if empty it will fetch the requested ID, else the page uri
	 * @param bool $paged Return current page URL without pagination
	 * @param bool $from_option Get the canonical uri option
	 *
	 * @since 2.0.0
	 */
	public function the_url( $url = '', $page_id = '', $paged = false, $from_option = true ) {

		if ( empty( $url ) ) {
			global $wp;

			/**
			 * Get url from options
			 * @since 2.2.9
			 */
			if ( $from_option ) {
				$url = $this->get_custom_field( '_genesis_canonical_uri' ) ? $this->get_custom_field( '_genesis_canonical_uri' ) : '';

				$genesis = $this->is_theme( 'genesis' );

				//* Genesis fallback
				if ( empty( $url ) && $genesis ) {
					$flag = $this->get_custom_field( 'saved_flag' ) ? true : false;

					if ( ! $flag )
						$url = genesis_get_custom_field( '_genesis_canonical_uri' ) ? genesis_get_custom_field( '_genesis_canonical_uri' ) : '';
				}
			}

			/**
			 * Fetch post object
			 * @since 2.2.4
			 */
			$post = get_post( $page_id, OBJECT );

			/**
			 * Trailing slash the post, or not.
			 * @since 2.2.4
			 */
			$slashit = true;

			/**
			 * Get page uri if Page ID is given
			 * @since 2.1.9
			 *
			 * Get page uri if $paged is true.
			 * Don't get page uri if there's no pretty permalinks or unconditioned post statusses
			 * @since 2.2.4
			 *
			 * Get relative full path.
			 * @since 2.3.0
			 */
			if ( isset( $post ) ) {
				if ( ( !empty( $page_id ) || $paged ) && ( '' != get_option( 'permalink_structure' ) && ! in_array( $post->post_status, array( 'draft', 'pending', 'auto-draft', 'future' ) ) ) ) {
					if ( !empty( $page_id ) && $page_id == get_option( 'page_on_front' ) && !$paged ) {
						$path = '';
					} else {
						$path = $this->get_relative_url( $post );
					}
				} else if ( '' == get_option( 'permalink_structure' ) || in_array( $post->post_status, array( 'draft', 'pending', 'auto-draft', 'future' ) ) ) {
					if ( !empty( $post->ID ) && $post->ID != get_option( 'page_on_front' ) ) {
						$path = $this->the_url_path_default_permalink_structure( $post );
						$slashit = false;
					} else {
						// Probably home page
						$path = '';
					}
				} else if ( !empty( $page_id ) || $paged ) {
					$path = $this->get_relative_url( $post );
				} else {
					$path = $wp->request;
				}
			}

			//* Generate URL
			if ( empty( $url ) ) {

				$path = isset( $path ) ? $path : '';

				//* Domain Mapping canonical url
				$url = $this->the_url_wpmudev_domainmap( $path );

				//* Non-domainmap URL
				if ( empty( $url ) ) {
				//	$url = home_url( add_query_arg( array(), $path ) );
					$url = home_url( $path );
					$scheme = is_ssl() ? 'https' : 'http';
				}
			}
		}

		$scheme = isset( $scheme ) ? $scheme : '';

		$output = esc_url( $url, $scheme );

		/**
		 * Slash it only if $slashit is true
		 *
		 * @since 2.2.4
		 */
		if ( $slashit )
			$output = trailingslashit( $output );

		return $output;
	}

	/**
	 * Generates relative URL for current post_ID.
	 *
	 * @param object $post The post.
	 *
	 * @since 2.3.0
	 */
	public function get_relative_url( $post ) {

		$home_url = home_url();
		$permalink = get_permalink( $post->ID );
		$count = 1;

		$path = str_replace( $home_url, '', $permalink, $count );

		return $path;
	}

	/**
	 * Creates canonical url for the default permalink structure.
	 *
	 * @param object $post The post.
	 *
	 * @since 2.3.0
	 */
	public function the_url_path_default_permalink_structure( $post ) {

		if ( !is_singular() ) {
			//* We're on a taxonomy
			$object = get_queried_object();

			if ( is_object( $object ) ) {
				if ( is_category() ) {
					$id = $object->term_id;
					$path = '?cat=' . $id;
				} else if ( is_tag() ) {
					$name = $object->name;
					$path = '?tag=' . $id;
				} else if ( is_date() ) {
					global $wp_query;

					$query = $wp_query->query;

					$year = $query->year;
					$month = !empty( $query->monthnum ) ? '&monthnum=' . $query->monthnum : '';

					$path = '?year=' . $year . $month;
				} else if ( is_author() ) {
					$name = $object->author_name;
					$path = '?author=' . $name;
				} else if ( is_tax() ) {
					$name = $object->taxonomy;
					$path = '?taxonomy=' . $name;
				} else {
					$id = $object->ID;
					$path = '?p=' . $id;
				}
			} else {
				$id = $post->ID;
				$path = '?p=' . $id;
			}
		} else {
			$id = $post->ID;
			$path = '?p=' . $id;
		}

		return $path;
	}

	/**
	 * Try to get an canonical URL when WPMUdev Domain Mapping is active.
	 *
	 * @param string $path The post relative path.
	 *
	 * @since 2.3.0
	 */
	public function the_url_wpmudev_domainmap( $path ) {

		if ( $this->is_domainmapping_active() ) {
			global $wpdb,$blog_id;

			//* Check if the domain is mapped
			$mapped_domain = wp_cache_get( 'wap_mapped_domain_' . $blog_id, 'domain_mapping' );
			if ( false === $mapped_domain ) {
				$mapped_domain = $wpdb->get_var( $wpdb->prepare( "SELECT domain FROM {$wpdb->base_prefix}domain_mapping WHERE blog_id = %d", $blog_id ) ); //string
				wp_cache_set( 'wap_mapped_domain_' . $blog_id, $mapped_domain, 'domain_mapping', 3600 ); // 1 hour
			}

			if ( !empty( $mapped_domain ) ) {

				//* Fetch scheme
				$mappedscheme = wp_cache_get('wap_mapped_scheme_' . $blog_id, 'domain_mapping' );
				if ( false === $mappedscheme ) {
					$mappedscheme = $wpdb->get_var( $wpdb->prepare( "SELECT scheme FROM {$wpdb->base_prefix}domain_mapping WHERE blog_id = %d", $blog_id ) ); //bool
					wp_cache_set('wap_mapped_scheme_' . $blog_id, $mappedscheme, 'domain_mapping', 3600 ); // 1 hour
				}

				if ( $mappedscheme === '1' ) {
					$scheme_full = 'https://';
					$scheme = 'https';
				} else if ( $mappedscheme === '0' ) {
					$scheme_full = 'http://';
					$scheme = 'http';
				}

				// Put it all together
				return $url = trailingslashit( $scheme_full . $mapped_domain ) . ltrim( $path, '\/' );
			}
		}

		return '';
	}

	/**
	 * Adds og:image
	 *
	 * @uses get_header_image
	 *
	 * @param string $post_id 	the post ID
	 * @param string $image		output url for image
	 *
	 * @filter hmpl_og_image_args 	: @param image the image url
	 *								: @param override always use the set url
	 *								: @param frontpage always use the set url on the front page
	 * The image set in the filter will always be used as fallback
	 *
	 * @since 2.2.1
	 *
	 * @todo create options and upload area
	 */
	public function get_image( $post_id = '', $args = array() ) {

		if ( empty( $post_id ) )
			$post_id = get_the_ID();

		if ( empty( $post_id ) )
			return '';

		$defaults = array(
				'post_id'	=> $post_id,
				'size'		=> 'full',
				'icon'		=> 0,
				'attr'		=> '',
				'image'		=> '',
				'override'	=> false,
				'frontpage'	=> true,
			);

		/**
		 * @since 2.0.5
		 */
		//* Marked for removal @since 2.3.5
		$defaults = apply_filters( 'hmpl_og_image_args', $defaults, $args );
		//* @since 2.3.0 filter.
		$defaults = (array) apply_filters( 'the_seo_framework_og_image_args', $defaults, $args );

		$args = wp_parse_args( $args, $defaults );

		$image = $args['image'];

		$is_front = $post_id == get_option( 'page_on_front' ) ? true : false;

		/**
		 * Fetch image if
		 * + no image found (always go)
		 * + override is true (always go)
		 * + is not front page AND frontpage override is set to true
		 */
		if ( empty( $image ) || $args['override'] !== false || ( !$is_front && $args['frontpage'] !== false ) )
			$image = $this->get_image_from_post_thumbnail( $args );

		//* Get the WP 4.3.0 Site Icon
		if ( empty( $image ) )
			$image = $this->site_icon();

		//* Fallback: Get header image if exists
		if ( empty( $image ) )
			$image = get_header_image();

		//* If there still is no image, get the "site avatar" from WPMUdev Avatars
		if ( empty( $image ) )
			$image = $this->get_image_from_wpmudev_avatars();

		return $image;
	}

	/**
	 * Fetches image from post thumbnail.
	 * Resizes the image between 1500px if bigger. Then it saves the image and
	 * Keeps dimensions relative.
	 *
	 * @param array $args Image arguments.
	 *
	 * @since 2.3.0
	 *
	 * @return string|null the image url.
	 */
	public function get_image_from_post_thumbnail( $args ) {

		$image = '';

		if ( has_post_thumbnail( $args['post_id'] ) ) {
			$id = get_post_thumbnail_id( $args['post_id'] );
			$src = wp_get_attachment_image_src( $id, $args['size'], $args['icon'], $args['attr'] );

			$i = $src[0];
			$w = $src[1];
			$h = $src[2];

			//* Prefered 1500px, resize it
			if ( $w > 1500 || $h > 1500 ) {

				//* Square
				if ( $w == $h ) {
					$w = 1500;
					$h = 1500;

				//* Landscape
				} elseif ( $w > $h ) {
					$dev = $w / 1500;

					$h = $h / $dev;

					$h = round( $h );
					$w = 1500;

				//* Portrait
				} elseif ( $h > $w ) {
					$dev = $h / 1500;

					$w = $w / $dev;

					$w = round( $w );
					$h = 1500;
				}

				// Get path of image and load it into the wp_get_image_editor
				$i_file_path = get_attached_file( $id );

				$i_file_old_name	= basename(get_attached_file( $id ));
				$i_file_ext			= pathinfo($i_file_path, PATHINFO_EXTENSION );

				if ( !empty( $i_file_ext ) ) {
					$i_file_dir_name 	= pathinfo($i_file_path, PATHINFO_DIRNAME );
					// Add trailing slash
					$i_file_dir_name	.= ( substr( $i_file_dir_name, -1 ) == '/' ? '' : '/' );

					$i_file_file_name 	= pathinfo( $i_file_path, PATHINFO_FILENAME );

					// Yes I know, I should use generate_filename, but it's slower.
					// Will look at that later. This is already 100 lines of correctly working code.
					$new_image_dirfile 	= $i_file_dir_name . $i_file_file_name . '-' . $w . 'x' . $h . '.' . $i_file_ext;

					// This should work on multisite too.
					$upload_dir 	= wp_upload_dir();
					$upload_url 	= $upload_dir['baseurl'];
					$upload_basedir = $upload_dir['basedir'];

					// Dub this $new_image
					$new_image_url = preg_replace( '/' . preg_quote( $upload_basedir, '/') . '/', $upload_url, $new_image_dirfile );

					// Generate file if it doesn't exists yet.
					if ( !file_exists( $new_image_dirfile ) ) {

						$image_editor = wp_get_image_editor( $i_file_path );

						if ( ! is_wp_error( $image_editor ) ) {
							$image_editor->resize( $w, $h, false );
							$image_editor->set_quality( 60 ); // Let's save some bandwidth, Facebook compresses it even further anyway.
							$image_editor->save( $new_image_dirfile );
						}
					}

					$i = $new_image_url;
				}
			}

			$image = $i;
		}

		return $image;
	}

	/**
	 * Fetches site image from WPMUdev Avatars.
	 *
	 * @since 2.3.0
	 *
	 * @return string|null the image url.
	 */
	public function get_image_from_wpmudev_avatars() {

		$image = '';

		if ( function_exists( 'is_plugin_active' ) && is_plugin_active( 'avatars/avatars.php' ) ) {
			global $ms_avatar,$blog_id;

			$size = '256';

			$file = $ms_avatar->blog_avatar_dir . $ms_avatar->encode_avatar_folder( $blog_id ) . '/blog-' . $blog_id . '-' . $size . '.png';
			$path = '';

			if ( is_file( $file ) ) {
				$upload_dir = wp_upload_dir();
				$upload_url = $upload_dir['baseurl'];
			//	$debug = '0';

				// Isn't there a more elegant core option? =/
				// I'm basically backwards enginering the wp_upload_dir
				// function to get the base url without /sites/blogid or /blogid.
				if ( is_multisite() && ! ( is_main_network() && is_main_site() && defined( 'MULTISITE' ) ) ) {
					if ( ! get_site_option( 'ms_files_rewriting' ) ) {
						if ( defined( 'MULTISITE' ) ) {
							$upload_url = str_replace( '/sites/' . $blog_id, '', $upload_url );
						} else {
							// This should never run.
							$upload_url = str_replace( '/' . $blog_id, '', $upload_url );
						}
					} else if ( defined( 'UPLOADS' ) && ! ms_is_switched() ) {
						// Special cases.
						// Where UPLOADS is defined AND we're on the main blog AND
						// WPMUdev avatars is used AND file is uploaded on main blog AND
						// no header image is set AND no favicon is uploaded.
						//
						// So yeah: I'm not sure what to do here so I'm just gonna fall back to default.
						// I'll wait for a bug report.
						$upload_url = str_replace( '/sites/' . $blog_id, '', $upload_url );
					}
				}

				// I think I should've used get_site_url...
				$avatars_url = trailingslashit( trailingslashit( $upload_url ) . basename( dirname( $ms_avatar->blog_avatar_dir ) ) );
				$path = preg_replace( '/' . preg_quote( dirname( $ms_avatar->blog_avatar_dir ) . '/', '/') . '/', $avatars_url, $file );
			}
			$image = !empty( $path ) ? $path : '';
		}

		return $image;
	}

	/**
	 * Fetches site icon brought in WordPress 4.3.0
	 *
	 * @param string $size 	The icon size, accepts 'full' and pixel values
	 * @since 2.2.1
	 *
	 * @return string url site icon, not escaped.
	 */
	public function site_icon( $size = 'full' ) {

		$icon = '';

		if ( $this->wp_version( '4.3.0', '>=' ) && function_exists( 'has_site_icon' ) ) {
			if ( $size == 'full' ) {
				$site_icon_id = get_option( 'site_icon' );

				$url_data = '';

				if ( $site_icon_id ) {
					$url_data = wp_get_attachment_image_src( $site_icon_id, $size );
				}

				$icon = $url_data ? $url_data[0] : '';
			} else if ( abs( $size ) ) {
				$icon = get_site_icon_url( $size );
			}
		}
		return $icon;
	}

	/**
	 * Output the `index`, `follow`, `noodp`, `noydir`, `noarchive` robots meta code in array
	 *
	 * @since 2.2.2
	 *
	 * @uses genesis_get_seo_option()   Get SEO setting value.
	 * @uses genesis_get_custom_field() Get custom field value.
	 *
	 * @global WP_Query $wp_query Query object.
	 *
	 * @return array|null robots
	 */
	public function robots_meta() {
		global $wp_query;

		$query_vars = is_object( $wp_query ) ? (array) $wp_query->query_vars : '';
		$paged = is_array( $query_vars ) ? $query_vars["paged"] : '';

		//* Genesis only check
		$genesis = $this->is_theme( 'genesis' );

		//* Defaults
		$meta = array(
			'noindex'   => $this->get_option( 'site_noindex' ) ? 'noindex' : '',
			'nofollow'  => $this->get_option( 'site_nofollow' ) ? 'nofollow' : '',
			'noarchive' => $this->get_option( 'site_noarchive' ) ? 'noarchive' : '',
			'noodp'     => $this->get_option( 'noodp' ) ? 'noodp' : '',
			'noydir'    => $this->get_option( 'noydir' ) ? 'noydir' : '',
		);

		/**
		 * Check the Robots SEO settings, set noindex for paged archives.
		 * @since 2.2.4
		 */
		if ( (int) $paged > (int) abs(1) ) {
			$meta['noindex'] = $this->get_option( 'paged_noindex' ) ? 'noindex' : $meta['noindex'];
		}

		/**
		 * Check if archive is empty, set noindex for those.
		 *
		 * @todo maybe create option
		 * @since 2.2.8
		 */
		if ( isset( $wp_query->post_count ) && $wp_query->post_count === (int) 0 ) {
			$meta['noindex'] = 'noindex';
		}

		//* Check home page SEO settings, set noindex, nofollow and noarchive
		if ( is_front_page() ) {
			$meta['noindex']   = $this->get_option( 'homepage_noindex' ) ? 'noindex' : $meta['noindex'];
			$meta['nofollow']  = $this->get_option( 'homepage_nofollow' ) ? 'nofollow' : $meta['nofollow'];
			$meta['noarchive'] = $this->get_option( 'homepage_noarchive' ) ? 'noarchive' : $meta['noarchive'];
		}

		if ( is_category() ) {
			$term = $wp_query->get_queried_object();

			$meta['noindex']   = $term->admeta['noindex'] ? 'noindex' : $meta['noindex'];
			$meta['nofollow']  = $term->admeta['nofollow'] ? 'nofollow' : $meta['nofollow'];
			$meta['noarchive'] = $term->admeta['noarchive'] ? 'noarchive' : $meta['noarchive'];

			if ( empty( $meta['noindex'] ) )
				$meta['noindex'] = $this->get_option( 'category_noindex' ) ? 'noindex' : $meta['noindex'];

			if ( empty( $meta['nofollow'] ) )
				$meta['nofollow'] = $this->get_option( 'category_nofollow' ) ? 'nofollow' : $meta['nofollow'];

			if ( empty( $meta['noarchive'] ) )
				$meta['noarchive'] = $this->get_option( 'category_noindex' ) ? 'noarchive' : $meta['noarchive'];

			/* @since 2.3.0
			if ( $genesis ) {
				//* noindex paged archives, if canonical archives is off
				$paged = get_query_var( 'paged' ) ? get_query_var( 'paged' ) : 1;
				$meta['noindex'] = $paged > 1 && ! genesis_get_seo_option( 'canonical_archives' ) ? 'noindex' : $meta['noindex'];
			}
			*/
		}

		if ( is_tag() ) {
			$term = $wp_query->get_queried_object();

			$meta['noindex']   = $term->admeta['noindex'] ? 'noindex' : $meta['noindex'];
			$meta['nofollow']  = $term->admeta['nofollow'] ? 'nofollow' : $meta['nofollow'];
			$meta['noarchive'] = $term->admeta['noarchive'] ? 'noarchive' : $meta['noarchive'];

			if ( empty( $meta['noindex'] ) )
				$meta['noindex'] = $this->get_option( 'tag_noindex' ) ? 'noindex' : $meta['noindex'];

			if ( empty( $meta['nofollow'] ) )
				$meta['nofollow'] = $this->get_option( 'tag_nofollow' ) ? 'nofollow' : $meta['nofollow'];

			if ( empty( $meta['noarchive'] ) )
				$meta['noarchive'] = $this->get_option( 'tag_noindex' ) ? 'noarchive' : $meta['noarchive'];

			/* @since 2.3.0
			if ( $genesis && !$flag ) {
				//* noindex paged archives, if canonical archives is off
				$paged = get_query_var( 'paged' ) ? get_query_var( 'paged' ) : 1;
				$meta['noindex'] = $paged > 1 && ! genesis_get_seo_option( 'canonical_archives' ) ? 'noindex' : $meta['noindex'];
			}
			*/
		}

		// Is custom Taxonomy page. But not a category or tag. Should've recieved specific term SEO settings.
		if ( is_tax() ) {
			$term = get_term_by( 'slug', get_query_var( 'term' ), get_query_var( 'taxonomy' ) );

			$flag = $term->admeta['saved_flag'];

			$meta['noindex']   = $term->admeta['noindex'] ? 'noindex' : $meta['noindex'];
			$meta['nofollow']  = $term->admeta['nofollow'] ? 'nofollow' : $meta['nofollow'];
			$meta['noarchive'] = $term->admeta['noarchive'] ? 'noarchive' : $meta['noarchive'];

			/* @since 2.3.0
			if ( $genesis && !$flag ) {
				//* noindex paged archives, if canonical archives is off
				$paged = get_query_var( 'paged' ) ? get_query_var( 'paged' ) : 1;
				$meta['noindex'] = $paged > 1 && ! genesis_get_seo_option( 'canonical_archives' ) ? 'noindex' : $meta['noindex'];
			}
			*/
		}

		/* @since 2.3.0
		if ( $genesis ) {
			if ( is_post_type_archive() && genesis_has_post_type_archive_support() ) {
				$meta['noindex']   = genesis_get_cpt_option( 'noindex' ) ? 'noindex' : $meta['noindex'];
				$meta['nofollow']  = genesis_get_cpt_option( 'nofollow' ) ? 'nofollow' : $meta['nofollow'];
				$meta['noarchive'] = genesis_get_cpt_option( 'noarchive' ) ? 'noarchive' : $meta['noarchive'];

				//* noindex paged archives, if canonical archives is off
				$paged = get_query_var( 'paged' ) ? get_query_var( 'paged' ) : 1;
				$meta['noindex'] = $paged > 1 && ! genesis_get_seo_option( 'canonical_archives' ) ? 'noindex' : $meta['noindex'];
			}
		}
		*/

		if ( is_author() ) {
			$flag = get_the_author_meta( 'saved_flag', (int) get_query_var( 'author' ) ) ? true : false;

			$meta['noindex']   = get_the_author_meta( 'noindex', (int) get_query_var( 'author' ) ) ? 'noindex' : $meta['noindex'];
			$meta['nofollow']  = get_the_author_meta( 'nofollow', (int) get_query_var( 'author' ) ) ? 'nofollow' : $meta['nofollow'];
			$meta['noarchive'] = get_the_author_meta( 'noarchive', (int) get_query_var( 'author' ) ) ? 'noarchive' : $meta['noarchive'];

			if ( empty( $meta['noindex'] ) )
				$meta['noindex'] = $this->get_option( 'author_noindex' ) ? 'noindex' : $meta['noindex'];

			if ( empty( $meta['nofollow'] ) )
				$meta['nofollow'] = $this->get_option( 'author_nofollow' ) ? 'nofollow' : $meta['nofollow'];

			if ( empty( $meta['noarchive'] ) )
				$meta['noarchive'] = $this->get_option( 'author_noarchive' ) ? 'noarchive' : $meta['noarchive'];

			/* @since 2.3.0
			if ( $genesis && !$flag ) {
				//* noindex paged archives, if canonical archives is off
				$paged = get_query_var( 'paged' ) ? get_query_var( 'paged' ) : 1;
				$meta['noindex'] = $paged > 1 && ! genesis_get_seo_option( 'canonical_archives' ) ? 'noindex' : $meta['noindex'];
			}
			*/
		}

		if ( is_date() ) {
			$meta['noindex'] = $this->get_option( 'date_noindex' ) ? 'noindex' : $meta['noindex'];
			$meta['nofollow'] = $this->get_option( 'date_nofollow' ) ? 'nofollow' : $meta['nofollow'];
			$meta['noarchive'] = $this->get_option( 'date_noarchive' ) ? 'noarchive' : $meta['noarchive'];
		}

		if ( is_search() ) {
			$meta['noindex'] = $this->get_option( 'search_noindex' ) ? 'noindex' : $meta['noindex'];
			$meta['nofollow'] = $this->get_option( 'search_nofollow' ) ? 'nofollow' : $meta['nofollow'];
			$meta['noarchive'] = $this->get_option( 'search_noarchive' ) ? 'noarchive' : $meta['noarchive'];
		}

		if ( is_singular() ) {
			$flag = $this->get_custom_field( 'saved_flag' ) ? true : false;

			$meta['noindex'] = $this->get_custom_field( '_genesis_noindex' ) ? 'noindex' : $meta['noindex'];
			if ( $genesis && empty($meta['noindex']) && !$flag )
				$meta['noindex'] = genesis_get_custom_field( '_genesis_noindex' ) ? 'noindex' : $meta['noindex'];

			$meta['nofollow'] = $this->get_custom_field( '_genesis_nofollow' ) ? 'nofollow' : $meta['nofollow'];
			if ( $genesis && empty($meta['nofollow']) && !$flag )
				$meta['nofollow'] = genesis_get_custom_field( '_genesis_nofollow' ) ? 'nofollow' : $meta['nofollow'];

			$meta['noarchive'] = $this->get_custom_field( '_genesis_noarchive' ) ? 'noarchive' : $meta['noarchive'];
			if ( $genesis && empty($meta['noarchive']) && !$flag )
				$meta['noarchive'] = genesis_get_custom_field( '_genesis_noarchive' ) ? 'noarchive' : $meta['noarchive'];
		}

		//* Strip empty array items
		$meta = array_filter( $meta );

		return $meta;
	}

	/**
	 * Generates shortlink url
	 *
	 * @since 2.2.2
	 *
	 * @param int $post_id The post ID
	 * @return string|null Escaped site Shortlink URL
	 */
	public function get_shortlink( $post_id = 0 ) {

		if ( $this->get_option( 'shortlink_tag' ) ) {

			if ( is_singular() ) {
				$shortlink = wp_get_shortlink( $post_id, 'query' );
			} else if ( ! is_front_page() ) {
				$object = get_queried_object();

				if ( is_category() ) {
					$id = $object->term_id;
					$path = '?cat=' . $id;
				}

				if ( is_tag() ) {
					$name = $object->name;
					$path = '?tag=' . $name;
				}

				if ( is_date() ) {
					// This isn't exactly "short" for a shortlink...
					// @TODO find alternative? -> requires wp rewrite.
					$year = get_query_var( 'year' );
					$month = get_query_var( 'monthnum' ) ? '&monthnum=' . get_query_var( 'monthnum' ) : '';

					$path = '?year=' . $year . $month;
				}

				if ( is_author() ) {
					$id = $object->ID;
					$path = '?author=' . $id;
				}

				if ( is_tax() ) {
					$id = $object->ID;
					$path = '?taxonomy=' . $id;
				}

				if ( empty( $path ) ) {
					$id = $object->ID;
					$shortlink = wp_get_shortlink( $id, 'post' );
				}

				if ( !empty( $path ) )
					$shortlink = home_url( $path );

			} else if ( 'page' == get_option( 'show_on_front' ) && get_queried_object_id() == get_option( 'page_for_posts' ) ) {
				//* Page for posts
				$id =  get_queried_object_id();
				$shortlink = wp_get_shortlink( $id, 'post' );
			} else {
				//* Home page
				$shortlink = home_url( '/' );
			}

			return esc_url_raw( $shortlink );
		}

		return '';
	}

	/**
	 * Generates Previous and Next links
	 *
	 * @since 2.2.4
	 *
	 * @param string $prev_next Previous or next page link
	 * @param int $post_id The post ID
	 *
	 * @return string|null Escaped site Pagination URL
	 */
	public function get_paged_url( $prev_next = 'next', $post_id = 0 ) {

		if ( !$this->get_option( 'prev_next_posts' ) && !$this->get_option( 'prev_next_archives' ) )
			return '';

		global $wp_query;

		$prev = '';
		$next = '';

		if ( ! is_singular() && $this->get_option( 'prev_next_archives' ) ) {

			$paged = get_query_var( 'paged' ) ? (int) get_query_var( 'paged' ) : 1;
		//	$paged = get_query_var( 'paged' ) ? intval( get_query_var( 'paged' ) ) : 1;

			if ( $prev_next == 'prev' )
				$prev = $paged > 1 ? get_previous_posts_page_link() : $prev;

			if ( $prev_next == 'next' )
				$next = $paged < $wp_query->max_num_pages ? get_next_posts_page_link() : $next;

		} else if ( is_singular() && $this->get_option( 'prev_next_posts' ) ) {

			$page  = (int) get_query_var( 'page' );
		//	$page  = intval( get_query_var( 'page' ) );

			$numpages = substr_count( $wp_query->post->post_content, '<!--nextpage-->' ) + 1;

			if ( $numpages && ! $page ) {
				$page = 1;
			}

			if ( $prev_next == 'prev' ) {
				if ( $page > 1 ) {
					$prev = $this->get_paged_post_url( $page - 1, $post_id, 'prev' );
				}
			}

			if ( $prev_next == 'next' ) {
				if ( $page < $numpages ) {
					$next = $this->get_paged_post_url( $page + 1, $post_id, 'next' );
				}
			}

		}

		if ( $prev )
			return esc_url_raw( $prev );

		if ( $next )
			return esc_url_raw( $next );

		return '';
	}

	/**
	 * Return the special URL of a paged post.
	 *
	 * Taken from _wp_link_page() in WordPress core, but instead of anchor markup, just return the URL.
	 * Also adds WPMUdev Domain Mapping support and is optimized for speed.
	 *
	 * @uses $this->the_url_from_cache();
	 * @since 2.2.4
	 *
	 * @param int $i The page number to generate the URL from.
	 * @param int $post_id The post ID
	 * @param string $pos Which url to get, accepts next|prev
	 *
	 * @return string Unescaped URL
	 */
	public function get_paged_post_url( $i, $post_id = 0, $pos = '' ) {

		$from_option = false;

		if ( abs(1) === $i ) {
			$url = $this->the_url_from_cache( '', $post_id, true, $from_option );
		} else {
			$post = get_post( $post_id );

			/**
			 * Fix the url.
			 *
			 * @since 2.2.5
			 */
			if ( $i >= 2 ) {
				//* Fix adding pagination url.

				$urlfromcache = $this->the_url_from_cache( '', '', '', $from_option );

				// Calculate current page number.
				$int_current = 'next' == $pos ? $i -1 : $i + 1;
				$string_current = (string) $int_current;

				if ( $i === (int) 1 ) {
					//* We're returning to the first page. Continue normal behaviour.
					$urlfromcache = $urlfromcache;
				} else {
					//* We're adding a page.
					$last_occurence = strrpos( $urlfromcache, '/' . $string_current . '/' );

					if ( $last_occurence !== false )
						$urlfromcache = substr_replace( $urlfromcache, '/', $last_occurence, strlen( '/' . $string_current . '/' ) );
				}
			} else {
				$urlfromcache = $this->the_url_from_cache( '', '', '', $from_option );
			}

			if ( '' == get_option( 'permalink_structure' ) || in_array( $post->post_status, array( 'draft', 'pending' ) ) ) {

				$url = add_query_arg( 'page', $i, $urlfromcache );

			} else if ( 'page' == get_option( 'show_on_front' ) && get_option( 'page_on_front' ) == $post->ID ) {
				global $wp_rewrite;

				$url = trailingslashit( $urlfromcache ) . user_trailingslashit( "$wp_rewrite->pagination_base/" . $i, 'single_paged' );
			} else {
				$url = trailingslashit( $urlfromcache ) . user_trailingslashit( $i, 'single_paged' );
			}
		}

		return $url;
	}

	/**
	 * Return LD+Json search helper.
	 *
	 * @since 2.2.8
	 *
	 * @return escaped LD+json search helper string.
	 */
	public function ld_json_search() {

		// @TODO Create option for output.

		$context = json_encode( 'http://schema.org' );
		$webtype = json_encode( 'WebSite' );
		$url = json_encode( esc_url( home_url( '/' ) ) );
		$name = json_encode( get_bloginfo( 'name', 'raw' ) );
		$alternatename = ''; // TODO
		$actiontype = json_encode( 'SearchAction' );

		// Remove trailing quote and add it back.
		$target = mb_substr( json_encode( esc_url( home_url( '/?s=' ) ) ), 0, -1 ) . '{search_term_string}"';

		$queryaction = json_encode( 'required name=search_term_string' );

		$json = sprintf( '{"@context":%s,"@type":%s,"url":%s,"name":%s,"potentialAction":{"@type":%s,"target":%s,"query-input":%s}}', $context, $webtype, $url, $name, $actiontype, $target, $queryaction );

		return $json;
	}

	/**
	 * Return LD+Json Knowledge Graph helper.
	 *
	 * @since 2.2.8
	 *
	 * @return null|escaped LD+json Knowledge Graph helper string.
	 */
	public function ld_json_knowledge() {

		if ( ! $this->get_option( 'knowledge_output' ) )
			return '';

		$json = '';

		$knowledge_name = $this->get_option( 'knowledge_type' ) ? $this->get_option( 'knowledge_type' ) : get_bloginfo( 'name', 'raw' );

		$context = json_encode( 'http://schema.org' );
		$type = json_encode( ucfirst( $this->get_option( 'knowledge_type' ) ) );
		$name = json_encode( $knowledge_name );
		$url = json_encode( esc_url( home_url( '/' ) ) );

		$logo = '';

		if ( $this->get_option( 'knowledge_type' ) == 'organization' && $this->get_option( 'knowledge_logo' ) ) {
			$icon = $this->site_icon();

			if ( $icon ) {
				$logourl = esc_url_raw( $icon );

				//* Add trailing comma
				$logo = '"logo":' . json_encode( $logourl ) . ',';
			}
		}

		/**
		 * Fetch option names
		 *
		 * @uses filter the_seo_framework_json_options
		 */
		$options = (array) apply_filters( 'the_seo_framework_json_options', array(
			'knowledge_facebook',
			'knowledge_twitter',
			'knowledge_gplus',
			'knowledge_instagram',
			'knowledge_youtube',
			'knowledge_linkedin',
			'knowledge_pinterest',
			'knowledge_soundcloud',
			'knowledge_tumblr',
		) );

		$sameurls = '';

		// Put the urls together from the options.
		if ( is_array( $options ) ) {
			foreach ( $options as $option ) {
				$comma = ',';

				if ( $this->get_option( $option ) )
					$sameurls .= json_encode( $this->get_option( $option ) ) . $comma;
			}
		}

		//* Remove trailing comma
		$sameurls = mb_substr( $sameurls, 0, -1 );

		if ( !empty( $sameurls ) )
			$json = sprintf( '{"@context":%s,"@type":%s,"name":%s,"url":%s,%s"sameAs":[%s]}', $context, $type, $name, $url, $logo, $sameurls );

		return $json;
	}

}
