<?php
/**
 * Class AutoDescription_DoingItRight
 *
 * Adds data in a column to edit.php and edit-tags.php
 * Shows you if you're doing the SEO right.
 *
 * @since 2.1.9
 */
class AutoDescription_DoingItRight extends AutoDescription_Search {

	/**
	 * Constructor, load parent constructor
	 *
	 * Initalizes columns
	 */
	public function __construct() {
		parent::__construct();

		add_action( 'admin_init', array( $this, 'init_columns') );
	}

	/**
	 * Initializes columns
	 *
	 * Applies filter 'hmpl_ad_column_support' : returns array of supported pages.
	 * Applies filter 'the_seo_framework_column_support' : returns array of supported pages.
	 *
	 * @param array $support_admin_pages the supported admin pages
	 *
	 * @since 2.1.9
	 */
	 public function init_columns() {

		//* Marked for removal @since 2.3.5
		$supported_post_types = apply_filters( 'hmpl_ad_column_support', array(
			'posts' => 'posts',
			'pages' => 'pages',
			'category' => 'edit-category',
			'post_tag'  => 'edit-post_tag',
		//  'link_category'  => 'edit-link-category',
		//  'galleries' => 'gallery_posts', // Not yet. @TODO
		));

		//* @since 2.3.0 filter.
		$supported_post_types = (array) apply_filters( 'the_seo_framework_column_support', $supported_post_types );

		if ( is_array ( $supported_post_types ) ) {
			foreach ( $supported_post_types as $slug => $type ) {
				add_action( "manage_{$type}_columns", array( $this, 'add_column' ), 10, 1 );
				add_action( "manage_{$slug}_custom_column", array( $this, 'seo_column' ), 10, 3 );
			}
		}
	 }

	/**
	 * Adds SEO column on edit.php
	 *
	 * Applies filter the_seo_framework_show_seo_column : Show the SEO column in edit.php
	 *
	 * @param $offset 	determines where the column should be placed. Prefered before comments, then data, then tags.
	 *					If neither found, it will add the column to the end.
	 *
	 * @since 2.1.9
	 * @return array $columns the column data
	 */
	public function add_column( $columns ) {

		//* Marked for removal @since 2.3.5
		$show_seo_column = apply_filters( 'hmpl_ad_show_seo_column', true );
		//* @since 2.3.0 filter.
		$show_seo_column = (bool) apply_filters( 'the_seo_framework_show_seo_column', $show_seo_column );

		if ( ! $show_seo_column )
			return $columns;

		$seocolumn = array( 'ad_seo' => 'SEO' );

		// Try comments
		$offset = array_search( 'comments', array_keys( $columns ) );

		// Try Count (posts) on taxonomies
		if ( empty($offset) )
			$offset = array_search( 'posts', array_keys( $columns ) );

		// Try date
		if ( empty($offset) )
			$offset = array_search( 'date', array_keys($columns) );
		// Try tags
		if ( empty($offset) )
			$offset = array_search( 'tags', array_keys( $columns ) );

		// I tried but found nothing
		if ( $offset === false || $offset === 0 || $offset === null ) {
			$columns = array_merge( $columns, $seocolumn );
		// My trials have been succesful, you shall pass
		} else {
			// Cache array before splice
			$columns_before = $columns;

			$columns = array_merge(
				array_splice( $columns, 0, $offset ),
				$seocolumn,
				array_splice( $columns_before, $offset )
			);
		}

		return $columns;
	}

	/**
	 * Adds SEO column to two to the left.
	 *
	 * @param string $column the current column    : If it's a taxonomy, this is empty
	 * @param int $post_id the post id             : If it's a taxonomy, this is the column name
	 * @param string $tax_id this is empty         : If it's a taxonomy, this is the taxonomy id
	 * @param string $status the status in html
	 *
	 * @since 2.1.9
	 * @return array $columns the column data
	 */
	public function seo_column( $column, $post_id, $tax_id = '' ) {

		$status = '';

		$type = get_post_type( $post_id );

		// It's a bug in WP? I'll report.
		// Reported: https://core.trac.wordpress.org/ticket/33521
		if ( !empty( $tax_id ) || ! $type ) {
			$column = $post_id;
			$post_id = $tax_id;

			$screen = get_current_screen();

			if ( isset( $screen) )
				$type = $screen->taxonomy;
		}

		if ( $column == 'ad_seo' )
			$status = $this->post_status( $post_id, $type, true );

		echo $status;
	}

	/**
	 * Renders post status. Caches the output.
	 *
	 * Applies filter the_seo_framework_seo_bar_squared : Make the SEO Bar squared.
	 *
	 * @param int $post_id The Post ID or taxonomy ID
	 * @param string $type Is fetched on edit.php, inpost, taxonomies, etc.
	 * @param bool $html return the status in html or string
	 *
	 * @todo document this further. It's been created within a day.
	 *
	 * @since 2.1.9
	 * @return string $content the post SEO status
	 */
	protected function post_status( $post_id = '', $type = 'inpost', $html = true ) {

		$content = '';

		if ( empty( $post_id ) )
			$post_id = get_the_ID();

		if ( $post_id !== false || $post_id !== null ) {

			if ( empty( $type ) )
				$type = get_post_type( $post_id );

			if ( $type == 'post' ) {
				$post = __( 'Post', 'autodescription' );
				$use_meta = false;
			} else if ( $type == 'page' ) {
				$post = __( 'Page', 'autodescription' );
				$use_meta = false;
			} else if ( $type == 'category' ) {
				$post = __( 'Category', 'autodescription' );
				$use_meta = true;
			} else if ( $type == 'post_tag' ) {
				$post = __( 'Tag', 'autodescription' );
				$use_meta = true;
			} else {
				// Fallback to Page as it is generic.
				$post = __( 'Page', 'autodescription' );
				$use_meta = false;
			}

			$post_low = strtolower( $post );

			$is_front_page = get_option( 'page_on_front' ) == $post_id ? true : false;

			$bad = 'ad-seo-bad';
			$okay = 'ad-seo-okay';
			$good = 'ad-seo-good';
			$unknown = 'ad-seo-unknown';

			$titlen_notice = '';
			$desclen_notice = '';
			$title_notice = '';
			$description_notice = '';
			$redirect_notice = '';
			$noindex_notice = '';
			$desc_too_many = '';

			$title_i18n = __( 'Title:', 'autodescription' );
			$description_i18n = __( 'Description:', 'autodescription' );
			$index_i18n = __( 'Index:', 'autodescription' );
			$follow_i18n = __( 'Follow:', 'autodescription' );
			$archive_i18n = __( 'Archive:', 'autodescription' );
			$redirect_i18n = __( 'Redirect:', 'autodescription' );

			$desc_gen = false;
			$tit_gen = false;

			if ( ! $use_meta ) {
				// $flag = $this->get_custom_field( 'saved_flag', $post_id ) ? true : false;

				$redirect = $this->get_custom_field( 'redirect' );
				$noindex = $this->get_custom_field( '_genesis_noindex' );

				if ( $is_front_page ) {
					$noindex = $this->get_option( 'homepage_noindex' ) ? $this->get_option( 'homepage_noindex' ) : $noindex;
				}

				$ad_125 = 'ad-12-5';
				$ad_100 = '';
			} else {
				$term = get_term_by( 'id', $post_id, $type, OBJECT );

				$ad_savedflag = isset( $term->admeta['saved_flag'] ) ? $term->admeta['saved_flag'] : false;
				$flag = $ad_savedflag ? true : false;

				$noindex = isset( $term->admeta['noindex'] ) ? $term->admeta['noindex'] : '';
				$redirect = ''; // We don't apply redirect on taxonomies (yet)

				//* Genesis data fetch
				$genesis = $this->is_theme( 'genesis' );
				if ( $genesis && !$flag ) {
					$noindex = ! empty ( $noindex ) ? $noindex : $term->meta['noindex'];
				}

				$ad_125 = 'ad-16';
				$ad_100 = 'ad-100';
			}

			/**
			 * @todo create plugin
			 */
			//* Marked for removal @since 2.3.5
			$square_it = apply_filters( 'hmpl_ad_seo_bar_squared', false );
			//* @since 2.3.0 filter.
			$square_it = (bool) apply_filters( 'the_seo_framework_seo_bar_squared', $square_it );

			$square = $square_it ? 'square' : '';

			// No redirect or noindex found, proceed.
			if ( empty( $redirect ) && empty( $noindex ) ) {

				if ( ! $use_meta ) {
					$generated = $this->get_custom_field( '_genesis_title' ) ? true : false;

					// Fetch the title normally.
					$title = $this->title();

					$description = $this->get_custom_field( '_genesis_description' ) ? $this->get_custom_field( '_genesis_description' ) : '';

					$nofollow = $this->get_custom_field( '_genesis_nofollow' );
					$noarchive = $this->get_custom_field( '_genesis_noarchive' );

					if ( $is_front_page ) {
						$generated = $this->get_option( 'homepage_title' ) ? true : $generated;

						// Fetch the title normally.
						$title = $this->title();

						$description = $this->get_option( 'homepage_description' ) ? $this->get_option( 'homepage_description' ) : $description;

						$nofollow = $this->get_option( 'homepage_nofollow' ) ? $this->get_option( 'homepage_nofollow' ) : $nofollow;
						$noarchive = $this->get_option( 'homepage_noarchive' ) ? $this->get_option( 'homepage_noarchive' ) : $noarchive;
					}

				} else {
					$generated = isset( $term->admeta['doctitle'] ) && $term->admeta['doctitle'] ? true : false;

					// Fetch the title normally.
					if ( $generated ) {
						//* Let's try not to fix the bloated function for now.
						$blogname = get_bloginfo( 'name', 'raw' );

						$title_seperator = $this->title_seperator;
						$sep_option = $this->get_field_value( 'title_seperator' );
						$sep = array_search( $sep_option, array_flip( $title_seperator ), false );

						// Order doesn't matter. It's only for length calculation purposes.
						$title = $blogname . " $sep " . $term->admeta['doctitle'];
					} else {
						$title = $this->title( '', '', '', $post_id, $type );
					}

					$description = isset( $term->admeta['description'] ) ? $term->admeta['description'] : '';

					$nofollow = isset( $term->admeta['nofollow'] ) ? $term->admeta['nofollow'] : '';
					$noarchive = isset( $term->admeta['noarchive'] ) ? $term->admeta['noarchive'] : '';

					//* Genesis data fetch
					$genesis = $this->is_theme( 'genesis' );
					if ( $genesis && !$flag ) {
						$title			= ! empty ( $title ) 		? $title		: $term->meta['doctitle'];
						$description	= ! empty ( $description )	? $description	: $term->meta['description'];
						$nofollow		= ! empty ( $nofollow )		? $nofollow		: $term->meta['nofollow'];
						$noarchive		= ! empty ( $noarchive )	? $noarchive	: $term->meta['noarchive'];
					}
				}

				$tit_len = mb_strlen( $title );
				$desc_len = mb_strlen( $description );

				if ( ! $generated ) {
					if ( $use_meta ) {
						$title = $this->title( '', '', '', $post_id, $type );
					} else if ( $is_front_page ) {
						$title = $this->title( '', '', '', '', '', true);
					} else {
						$title = $this->title();
					}
					$tit_len = mb_strlen( $title );
					$tit_gen = true;
				}

				if ( $desc_len == 0 ) {
					if ( ! $use_meta ) {
						$description = $this->generate_description();
					} else {
						$description = $this->generate_description( '', $post_id, $type );
					}

					$desc_len = mb_strlen( $description );
					$desc_gen = true;
				}

				$desc_word_count = array_count_values( str_word_count( strtolower( $description ), 1 ) );

				if ( is_array ( $desc_word_count ) ) {
					foreach ( $desc_word_count as $desc_word => $desc_word_count ) {
						if ( $desc_word_count >= 3 ) {
							$desc_too_many[] = array( $desc_word => $desc_word_count );
						}
					}
				}

				// Add starting space
				$generated = ' ' . _x( 'G', 'Generated', 'autodescription');
				// Add starting break. Yes it's being put inside an HTML attribute. Yes it's allowed. No this can't be put into the title attribute.
				$generated_notice = '<br />' . __( 'Generated: Automatically generated.', 'autodescription');

				$gen_t = $tit_gen ? $generated : '';
				$gen_d = $desc_gen ? $generated : '';

				$gen_t_notice = $tit_gen ? $generated_notice : '';
				$gen_d_notice = $desc_gen ? $generated_notice : '';

				if ( $tit_len < 25 ) {
					$titlen_notice = $title_i18n . ' ' . __( 'far too short.', 'autodescription' );
					$titlen_class = $bad;
				} else if ( $tit_len < 50 ) {
					$titlen_notice = $title_i18n . ' ' . __( 'too short.', 'autodescription' );
					$titlen_class = $okay;
				} else if ( $tit_len > 55 && $tit_len < 75 ) {
					$titlen_notice = $title_i18n . ' ' . __( 'too long.', 'autodescription' );
					$titlen_class = $okay;
				} else if ( $tit_len >= 75 ) {
					$titlen_notice = $title_i18n . ' ' . __( 'far too long.', 'autodescription' );
					$titlen_class = $bad;
				} else {
					$titlen_notice = $title_i18n . ' ' . __( 'good.', 'autodescription' );
					$titlen_class = $good;
				}

				if ( !empty( $desc_too_many ) && is_array( $desc_too_many ) ) {
					$desclen_notice = $description_i18n;

					$words_count = count($desc_too_many);
					$desclen_class = $words_count <= 1 ? $okay : $bad;

					$count = 1;

					foreach ( $desc_too_many as $key => $desc_array ) {
						foreach ( $desc_array as $desc_value => $desc_count ) {
							$desclen_notice .= ' ';
							$desclen_notice .= sprintf( __( '%s is used %d times.', 'autodescription' ), ucfirst( $desc_value ), $desc_count );

							if ( $words_count > 1 && $count != $words_count )
								$desclen_notice .= '<br />'; // Yes, <br /> is used inside an attribute. Allowed.

							$count++;
						}
					}
				} else if ( $desc_len < 100 ) {
					$desclen_notice = $description_i18n . ' ' . __( 'far too short.', 'autodescription' );
					$desclen_class = $bad;
				} else if ( $desc_len < 150 ) {
					$desclen_notice = $description_i18n . ' ' . __( 'too short.', 'autodescription' );
					$desclen_class = $okay;
				} else if ( $desc_len > 165 && $desc_len < 180 ) {
					$desclen_notice = $description_i18n . ' ' . __( 'too long.', 'autodescription' );
					$desclen_class = $okay;
				} else if ( $desc_len >= 180 ) {
					$desclen_notice = $description_i18n . ' ' . __( 'far too long.', 'autodescription' );
					$desclen_class = $bad;
				} else {
					$desclen_notice = $description_i18n . ' ' . __( 'good.', 'autodescription' );
					$desclen_class = $good;
				}

				$ind_notice = $index_i18n . ' ' . sprintf( __( "%s is being indexed.", 'autodescription' ), $post );
				$ind_class = $good;

				/**
				 * Get noindex site option
				 *
				 * @since 2.2.2
				 */
				if ( $this->get_option( 'site_noindex' ) ) {
					$ind_notice .= '<br />' . sprintf( __( "But you've disabled indexing for the whole site.", 'autodescription' ), $post );
					$ind_class = $unknown;
				}

				if ( ! get_option( 'blog_public' ) ) {
					$ind_notice .= '<br />' . sprintf( __( "But the blog isn't set to public. This means WordPress disencourages indexing.", 'autodescription' ), $post );
					$ind_class = $unknown;
				}

				/**
				 * Check if archive is empty, and therefor has set noindex for those.
				 *
				 * @since 2.2.8
				 */
				if ( $use_meta && isset( $term->count ) && $term->count === (int) 0 ) {
					$ind_notice .= '<br />' . sprintf( __( "But there are no posts in this %s. Therefor indexing has been disabled.", 'autodescription' ), $post );
					$ind_class = $unknown;
				}

				if ( empty( $nofollow ) ) {
					$fol_notice = $follow_i18n . ' ' . sprintf( __( '%s links are being followed.', 'autodescription' ), $post );
					$fol_class = $good;

					/**
					 * Get nofolow site option
					 *
					 * @since 2.2.2
					 */
					if ( $this->get_option( 'site_nofollow' ) ) {
						$fol_notice .= '<br />' . sprintf( __( "But you've disabled following of links for the whole site.", 'autodescription' ), $post );
						$fol_class = $unknown;
					}
				} else {
					$fol_notice = $follow_i18n . ' ' . sprintf( __( "%s links aren't being followed.", 'autodescription' ), $post );
					$fol_class = $unknown;

					if ( ! get_option( 'blog_public' ) ) {
						$fol_notice .= '<br />' . sprintf( __( "But the blog isn't set to public. This means WordPress allows the links to be followed regardless.", 'autodescription' ), $post );
					}
				}

				if ( empty( $noarchive ) ) {
					$arc_notice = $archive_i18n . ' ' . sprintf( __( 'Search Engine are allowed to archive this %s.', 'autodescription' ), $post_low );
					$arc_class = $good;

					/**
					 * Get noarchive site option
					 *
					 * @since 2.2.2
					 */
					if ( $this->get_option( 'site_noarchive' ) ) {
						$arc_notice .= '<br />' . sprintf( __( "But you've disabled archiving for the whole site.", 'autodescription' ), $post );
						$arc_class = $unknown;
					}

				} else {
					$arc_notice = $archive_i18n . ' ' . sprintf( __( "Search Engine aren't allowed to archive this %s.", 'autodescription' ), $post_low );
					$arc_class = $unknown;

					if ( ! get_option( 'blog_public' ) ) {
						$arc_notice .= '<br />' . sprintf( __( "But the blog isn't set to public. This means WordPress allows the blog to be archived regardless.", 'autodescription' ), $post );
					}
				}

				$red_notice = $redirect_i18n . ' ' . sprintf( __( "%s isn't being redirected.", 'autodescription' ), $post );
				$red_class = $good;

				if ( !empty( $titlen_notice ) )
					$title_notice		= '<span class="ad-sec-wrap ad-25">'
										. '<a href="#" onclick="return false;" class="' . $titlen_class . '"  data-desc="' . $titlen_notice . $gen_t_notice . '">' . _x( 'T', 'Title', 'autodescription') . $gen_t . '</a>'
										. '<span class="screen-reader-text">' . $titlen_notice . $gen_t_notice . '</span>'
										. '</span>'
										;

				if ( !empty( $desclen_notice ) )
					$description_notice	= '<span class="ad-sec-wrap ad-25">'
										. '<a href="#" onclick="return false;" class="' . $desclen_class . '" data-desc="' . $desclen_notice . $gen_d_notice . '">' . _x( 'D', 'Description', 'autodescription') . $gen_d . '</a>'
										. '<span class="screen-reader-text">' . $desclen_notice . $gen_d_notice . '</span>'
										. '</span>'
										;

					$index_notice		= '<span class="ad-sec-wrap ' . $ad_125 . '">'
										. '<a href="#" onclick="return false;" class="' . $ind_class . '" data-desc="' . $ind_notice . '">' . _x( 'I', 'no-Index', 'autodescription') . '</a>'
										. '<span class="screen-reader-text">' . $ind_notice . '</span>'
										. '</span>'
										;

				if ( !empty( $fol_notice ) )
					$follow_notice		= '<span class="ad-sec-wrap ' . $ad_125 . '">'
					 					. '<a href="#" onclick="return false;" class="' . $fol_class . '" data-desc="' . $fol_notice . '">' . _x( 'F', 'no-Follow', 'autodescription') . '</a>'
										. '<span class="screen-reader-text">' . $fol_notice . '</span>'
										. '</span>'
										;


				if ( !empty( $arc_notice ) )
					$archive_notice		= '<span class="ad-sec-wrap ' . $ad_125 . '">'
										. '<a href="#" onclick="return false;" class="' . $arc_class . '" data-desc="' . $arc_notice . '">' . _x( 'A', 'no-Archive', 'autodescription') . '</a>'
										. '<span class="screen-reader-text">' . $arc_notice . '</span>'
										. '</span>'
										;

				// No redirection on taxonomies (yet).
				if ( ! $use_meta ) {
					$redirect_notice	= '<span class="ad-sec-wrap ' . $ad_125 . '">'
										. '<a href="#" onclick="return false;" class="' . $red_class . '" data-desc="' . $red_notice . '">' . _x( 'R', 'Redirect', 'autodescription') . '</a>'
										. '<span class="screen-reader-text">' . $red_notice . '</span>'
										. '</span>'
										;
				} else {
					$redirect_notice 	= '';
				}

				$content = sprintf( '<span class="ad-seo clearfix ' . $ad_100 . ' ' . $square . '"><span class="ad-bar-wrap">%s %s %s %s %s %s</span></span>', $title_notice, $description_notice, $index_notice, $follow_notice, $archive_notice, $redirect_notice );

			// Redirect and noindex found, why bother showing SEO.
			} else if ( !empty( $redirect ) && !empty( $noindex ) ) {

				$red_notice = $redirect_i18n . ' ' . sprintf( __( "%s is being redirected. This means no SEO values have to be set.", 'autodescription' ), $post );
				$red_class = $unknown;

				$redirect_notice	= '<span class="ad-sec-wrap ad-50">'
									. '<a href="#" onclick="return false;" class="' . $red_class . '" data-desc="' . $red_notice . '">' . _x( 'R', 'Redirect', 'autodescription') . '</a>'
									. '<span class="screen-reader-text">' . $red_notice . '</span>'
									. '</span>'
									;

				$noi_notice = $index_i18n . ' ' . sprintf( __( "%s is not being indexed. This means no SEO values have to be set.", 'autodescription' ), $post );
				$noi_class = $unknown;

				$noindex_notice		= '<span class="ad-sec-wrap ad-50">'
				 					. '<a href="#" onclick="return false;" class="' . $noi_class . '" data-desc="' . $noi_notice . '">' . _x( 'I', 'no-Index', 'autodescription') . '</a>'
									. '<span class="screen-reader-text">' . $noi_notice . '</span>'
									. '</span>'
									;

				$content = sprintf( '<span class="ad-seo clearfix ' . $ad_100 . ' ' . $square . '"><span class="ad-bar-wrap">%s %s</span></span>', $redirect_notice, $noindex_notice );

			// Redirect found, why bother showing SEO info?
			} else if ( !empty( $redirect ) && empty( $noindex ) ) {

				$red_notice = $redirect_i18n . ' ' . sprintf( __( "%s is being redirected. This means no SEO values have to be set.", 'autodescription' ), $post );
				$red_class = $unknown;

				$redirect_notice	= '<span class="ad-sec-wrap ad-100">'
									. '<a href="#" onclick="return false;" class="' . $red_class . '" data-desc="' . $red_notice . '">' . _x( 'R', 'Redirect', 'autodescription') . '</a>'
									. '<span class="screen-reader-text">' . $red_notice . '</span>'
									. '</span>'
									;

				$content = sprintf( '<span class="ad-seo clearfix ' . $ad_100 . ' ' . $square . '"><span class="ad-bar-wrap">%s</span></span>', $redirect_notice );

			// Noindex found, why bother showing SEO info?
			} else if ( empty( $redirect ) && !empty( $noindex ) ) {

				$noi_notice = $index_i18n . ' ' . sprintf( __( "%s is not being indexed. This means no SEO values have to be set.", 'autodescription' ), $post );
				$noi_class = $unknown;

				$noindex_notice	= '<span class="ad-sec-wrap ad-100">'
								. '<a href="#" onclick="return false;" class="' . $noi_class . '" data-desc="' . $noi_notice . '">' . _x( 'I', 'no-Index', 'autodescription') . '</a>'
								. '<span class="screen-reader-text">' . $noi_notice . '</span>'
								. '</span>'
								;

				$content = sprintf( '<span class="ad-seo clearfix ' . $ad_100 . ' ' . $square . '"><span class="ad-bar-wrap">%s</span></span>', $noindex_notice );
			}

		} else {
			$content = '<span>' . __( 'Failed to fetch post ID.', 'autodescription' ) . '</span>';
		}

		return $content;
	}

}
