<?php
/**
 * Class AutoDescription_Detect
 *
 * Detects other plugins and themes
 * Returns booleans
 *
 * @since 2.1.6
 */
class AutoDescription_Detect extends AutoDescription_Render {

	/**
	 * Constructor, load parent constructor
	 */
	public function __construct() {
		parent::__construct();
	}

	/**
	 * Detect active plugin by constant, class or function existence.
	 *
	 * @since 1.3.0
	 *
	 * @param array $plugins Array of array for constants, classes and / or functions to check for plugin existence.
	 *
	 * @return boolean True if plugin exists or false if plugin constant, class or function not detected.
	 */
	public function detect_plugin( $plugins ) {

		//* Check for classes
		if ( isset( $plugins['classes'] ) ) {
			foreach ( $plugins['classes'] as $name ) {
				if ( class_exists( $name ) )
					return true;
					break;
			}
		}

		//* Check for functions
		if ( isset( $plugins['functions'] ) ) {
			foreach ( $plugins['functions'] as $name ) {
				if ( function_exists( $name ) )
					return true;
					break;
			}
		}

		//* Check for constants
		if ( isset( $plugins['constants'] ) ) {
			foreach ( $plugins['constants'] as $name ) {
				if ( defined( $name ) )
					return true;
					break;
			}
		}

		//* No class, function or constant found to exist
		return false;
	}

	/**
	 * Checks if the (parent) theme name is loaded
	 *
	 * @NOTE will return true if ANY of the array values matches.
	 *
	 * @param string|array $themes the current theme name
	 * @param bool $use_cache If set to false don't use cache.
	 *
	 * @since 2.1.0
	 *
	 * Added cache.
	 * @since 2.2.4
	 *
	 * @return null before registering caches if wp_get_theme function can't be found.
	 * @return bool is theme active.
	 */
	public function is_theme( $themes = null, $use_cache = true ) {

		if ( ! $use_cache ) {
			//* Don't use cache.

			$theme_parent = strtolower( wp_get_theme()->get('Template') );
			$theme_name = strtolower( wp_get_theme()->get('Name') );

			if ( is_string( $themes ) ) {
				$themes = strtolower( $themes );
				if ( $theme_parent == $themes || $theme_name == $themes )
					return true;
			} else if ( is_array( $themes ) ) {
				foreach ( $themes as $theme ) {
					$theme = strtolower( $theme );
					if ( $theme_parent == $theme || $theme_name == $theme ) {
						return true;
						break;
					}
				}
			}

			return false;
		}

		static $themes_cache = array();

		//* Check theme check cache
		if ( is_string( $themes ) && isset( $themes_cache[$themes] ) ) {
			$themes = strtolower( $themes );
			//* Theme check has been cached
			return $themes_cache[$themes];
		}

		if ( is_array( $themes ) ) {
			foreach ( $themes as $theme ) {
				$theme = strtolower( $theme );
				if ( isset( $themes_cache[$theme] ) && in_array( $themes_cache[$theme], $themes ) && $themes_cache[$theme] ) {
					// Feature is found and true
					return $themes_cache[$theme];
					break;
				}
			}
		}

		$theme_parent = strtolower( wp_get_theme()->get('Template') );
		$theme_name = strtolower( wp_get_theme()->get('Name') );

		if ( is_string( $themes ) ) {
			$themes = strtolower( $themes );
			if ( $theme_parent == $themes || $theme_name == $themes )
				$themes_cache[$themes] = true;
		} else if ( is_array( $themes ) ) {
			foreach ( $themes as $theme ) {
				$theme = strtolower( $theme );
				if ( $theme_parent == $theme || $theme_name == $theme ) {
					return $themes_cache[$theme] = true;
					break;
				} else {
					$themes_cache[$theme] = false;
				}
			}
			return $themes_cache[$theme];
		}

		//* The theme isn't active
		if ( is_string( $themes ) && ! isset( $themes_cache[$themes] ) )
			$themes_cache[$themes] = false;

		return $themes_cache[$themes];
	}

	/**
	 * SEO plugin detection
	 *
	 * @since 1.3.0
	 *
	 * @thanks StudioPress :)
	 *
	 * Added caching.
	 * @since 2.2.5
	 */
	public function detect_seo_plugins() {

		static $plugins = null;

		if ( !isset( $plugins ) ) {
			// Use this filter to adjust plugin tests.
			//* Marked for removal @since 2.3.5
			$plugins_check = apply_filters(
				'hmpl_ad_detect_seo_plugins',
				//* Add to this array to add new plugin checks.
				array(

					// Classes to detect.
					'classes' => array(
						'All_in_One_SEO_Pack',
						'All_in_One_SEO_Pack_p',
						'HeadSpace_Plugin',
						'Platinum_SEO_Pack',
						'wpSEO',
						'SEO_Ultimate',
					),

					// Functions to detect.
					'functions' => array(),

					// Constants to detect.
					'constants' => array( 'WPSEO_VERSION', ),
				)
			);

			//* @since 2.3.0 filter.
			$plugins_check = (array) apply_filters( 'the_seo_framework_detect_seo_plugins', $plugins_check );

			$plugins = $this->detect_plugin( $plugins_check );

		}

		return $plugins;
	}

	/**
	 * Detects if plugins outputting og:type exists
	 *
	 * @note isn't used in $this->og_image() Because og:image may be output multiple times.
	 *
	 * @uses $this->detect_plugin()
	 *
	 * @since 1.3.0
	 * @return bool true if exists, false if not.
	 *
	 * Added caching.
	 * @since 2.2.5
	 */
	public function has_og_plugin() {

		static $has_plugin = null;

		if ( isset( $has_plugin ) )
			return $has_plugin;

		$plugins = array( 'classes' => array( 'WPSEO_OpenGraph', 'All_in_One_SEO_Pack_Opengraph' ) );

		if ( $this->detect_plugin( $plugins ) ) {
			$has_plugin = true;
		} else {
			$has_plugin = false;
		}

		return $has_plugin;
	}

	/**
	 * Detects if plugins outputting ld+json exists
	 *
	 * @uses $this->detect_plugin()
	 *
	 * @since 1.3.0
	 *
	 * @return bool
	 *
	 * Added caching.
	 * @since 2.2.5
	 */
	public function has_json_ld_plugin() {

		static $has_plugin = null;

		if ( isset( $has_plugin ) )
			return $has_plugin;

		$plugins = array('classes' => array( 'WPSEO_JSON_LD' ) );

		if ( $this->detect_plugin( $plugins ) ) {
			$has_plugin = true;
		} else {
			$has_plugin = false;
		}

		return $has_plugin;
	}

	/**
	 * Detecs sitemap plugins
	 *
	 * @uses $this->detect_plugin()
	 *
	 * @since 2.1.0
	 *
	 * @return bool
	 *
	 * Added caching.
	 * @since 2.2.5
	 */
	public function has_sitemap_plugin() {

		static $has_plugin = null;

		if ( isset( $has_plugin ) )
			return $has_plugin;

		$plugins = array(
				'classes' => array( 'GoogleSitemapGeneratorLoader', 'xml_sitemaps', 'All_in_One_SEO_Pack_Sitemap', 'SimpleWpSitemap', 'Incsub_SimpleSitemaps' ),
				'functions' => array( 'wpss_init', 'gglstmp_sitemapcreate' ),
			);

		if ( $this->detect_plugin( $plugins ) ) {
			$has_plugin = true;
		} else {
			$has_plugin = false;
		}

		return $has_plugin;
	}

	/**
	 * Determines if WP is above or below a version
	 *
	 * @since 2.2.1
	 *
	 * @param string $version the three part version to compare to WordPress
	 * @param string $compare the comparing operator, default "$version >= Current WP Version"
	 *
	 * @return bool wp version is "compare" to
	 */
	public function wp_version( $version = '4.3.0', $compare = '>=' ) {
		global $wp_version;

		$count = abs(0);

		foreach ( count_chars($wp_version, 1) as $val => $i ) {
			// Count each dot.
			if ( chr($val) == '.') {
				$count = $i;
				break;
			}
		}

		// Add a .0 if WP outputs something like 4.3 instead of 4.3.0
		if ( $count === abs(1) ) {
			$wp_version = $wp_version . '.0';
		}

		if ( version_compare( $wp_version, $version, $compare ) )
			return true;

		return false;
	}

	/**
	 * Checks for current theme support.
	 *
	 * Also, if it's cached as true from an array, it will be cached as string as well.
	 * This is desired.
	 *
	 * @NOTE will return true if ANY of the array values matches.
	 *
	 * @since 2.2.5
	 *
	 * @param string|array required $feature The features to check for.
	 * @param bool $use_cache If set to false don't use cache.
	 *
	 * @return bool theme support.
	 */
	public function detect_theme_support( $features, $use_cache = true ) {

		if ( ! $use_cache ) {
			//* Don't use cache.

			if ( is_string( $features ) && ( $this->current_theme_supports( $features ) ) )
				return true;

			if ( is_array( $features ) ) {
				foreach ( $features as $feature ) {
					if ( $this->current_theme_supports( $feature ) ) {
						return true;
						break;
					}
				}
			}

			return false;
		}

		//* Setup cache.
		static $cache = array();

		//* Check theme support cache
		if ( is_string( $features ) && isset( $cache[$features] ) )
			//* Feature support check has been cached
			return $cache[$features];

		//* Check theme support array cache
		if ( is_array( $features ) ) {
			foreach ( $features as $feature ) {
				if ( isset( $cache[$feature] ) && in_array( $cache[$feature], $features ) && $cache[$feature] ) {
					// Feature is found and true
					return $cache[$feature];
					break;
				}
			}
		}

		//* Setup cache values
		if ( is_string( $features ) ) {
			if ( $this->current_theme_supports( $features ) ) {
				return $cache[$features] = true;
			} else {
				return $cache[$features] = false;
			}
		} else if ( is_array( $features ) ) {
			foreach ( $features as $feature ) {
				if ( $this->current_theme_supports( $feature ) ) {
					return $cache[$feature] = true;
					break;
				} else {
					$cache[$feature] = false;
				}
			}
			return $cache[$feature];
		}

		// No true value found so far, let's return false.
		if ( ! isset( $cache[$features] ) )
			$cache[$features] = false;

		return $cache[$features];
	}

	/**
	 * Checks a theme's support for a given feature
	 *
	 * @since 2.2.5
	 *
	 * @global array $_wp_theme_features
	 *
	 * @param string $feature the feature being checked
	 * @return bool
	 *
	 * Taken from WP Core, but it now returns true on title-tag support.
	 */
	public function current_theme_supports( $feature ) {
		global $_wp_theme_features;

		if ( 'custom-header-uploads' == $feature )
			return $this->current_theme_supports( 'custom-header', 'uploads' );

		if ( !isset( $_wp_theme_features[$feature] ) )
			return false;

		if ( 'title-tag' == $feature ) {

			//* SEO Framework Edits. {

				//* The SEO Framework unique 'feature'.
				if ( true === $_wp_theme_features[$feature] )
					return true;

				//* We might as well return false now preventing the debug_backtrace();
				return false;

			//* } End SEO Framework Edits.

			// Don't confirm support unless called internally.
			$trace = debug_backtrace();

			if ( ! in_array( $trace[1]['function'], array( '_wp_render_title_tag', 'wp_title' ) ) ) {
				return false;
			}
		}

		// If no args passed then no extra checks need be performed
		if ( func_num_args() <= 1 )
			return true;

		$args = array_slice( func_get_args(), 1 );

		switch ( $feature ) {
			case 'post-thumbnails':
				// post-thumbnails can be registered for only certain content/post types by passing
				// an array of types to add_theme_support(). If no array was passed, then
				// any type is accepted
				if ( true === $_wp_theme_features[$feature] )  // Registered for all types
					return true;
				$content_type = $args[0];
				return in_array( $content_type, $_wp_theme_features[$feature][0] );

			case 'html5':
			case 'post-formats':
				// specific post formats can be registered by passing an array of types to
				// add_theme_support()

				// Specific areas of HTML5 support *must* be passed via an array to add_theme_support()

				$type = $args[0];
				return in_array( $type, $_wp_theme_features[$feature][0] );

			case 'custom-header':
			case 'custom-background' :
				// specific custom header and background capabilities can be registered by passing
				// an array to add_theme_support()
				$header_support = $args[0];
				return ( isset( $_wp_theme_features[$feature][0][$header_support] ) && $_wp_theme_features[$feature][0][$header_support] );
		}

		/**
		 * Filter whether the current theme supports a specific feature.
		 *
		 * The dynamic portion of the hook name, `$feature`, refers to the specific theme
		 * feature. Possible values include 'post-formats', 'post-thumbnails', 'custom-background',
		 * 'custom-header', 'menus', 'automatic-feed-links', 'title-tag' and 'html5'.
		 *
		 * @since WP 3.4.0
		 *
		 * @param bool   true     Whether the current theme supports the given feature. Default true.
		 * @param array  $args    Array of arguments for the feature.
		 * @param string $feature The theme feature.
		 */
		return apply_filters( "current_theme_supports-{$feature}", true, $args, $_wp_theme_features[$feature] );
	}

	/**
	 * Add doing it wrong notice in the footer/error-log
	 *
	 * @since 2.2.5
	 *
	 * @return void
	 */
	public function title_doing_it_wrong() {

		// Prevent error log spam.
		static $no_spam = null;

		if ( isset( $no_spam ) )
			return;

		$version = '2.2.5';
		$version = $this->autodescription_version( $version );

		$example = esc_html( "<title><?php wp_title(''); ?></title>" );
		$example2 = esc_html( esc_url( "https://codex.wordpress.org/Title_Tag" ) );

		$message = "wp_title() should be called with only a single empty parameter in your header. Like exactly so: <br>\r\n$example\r\n<br>";
		$message2 = "Alternatively, you could add the title-tag theme support. This is recommended. Read how-to here: \r\n$example2\r\n<br>";

		_doing_it_wrong( 'wp_title', $message . $message2, $version );

		$no_spam = true;

		return;
	}

	/**
	 * Detect WPMUdev Domain Mapping plugin.
	 *
	 * @since 2.3.0
	 *
	 * @return bool false if Domain Mapping isn't active
	 */
	public function is_domainmapping_active() {

		static $active = null;

		if ( isset( $active ) )
			return $active;

		if ( function_exists( 'is_plugin_active' ) ) {
			if ( is_plugin_active( 'domain-mapping/domain-mapping.php' ) ) {
				$active = true;
			} else {
				$active = false;
			}

			return $active;
		} else {
			//* Notify developer
			$version = $this->autodescription_version( '2.3.0' );
			_doing_it_wrong( __FUNCTION__, 'was called too early.', $version );

			return false;
		}
	}

}
