<?php
/**
 * Class AutoDescription_AdminInit
 *
 * Initializes the plugin for the wp-admin screens.
 * Enqueues css and javascript.
 *
 * @since 2.1.6
 */
class AutoDescription_Admin_Init extends AutoDescription_Init {

	/**
	 * Constructor, load parent constructor
	 *
	 * Initalizes wp-admin functions
	 */
	public function __construct() {
		parent::__construct();

		add_action( 'admin_init', array( $this, 'post_state' ) );
		add_action( 'admin_init', array( $this, 'post_type_support' ) );

		/**
		 * Removed.
		 * @since 2.2.4
		 */
	//	add_action( 'after_setup_theme', array($this, 'admin_run'), 10);

		/**
		 * @since 2.2.4
		 */
		add_filter( 'genesis_detect_seo_plugins', array( $this, 'no_more_genesis_seo' ), 10 );

		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_javascript' ), 10 );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_css' ), 10 );

	}

	/**
	 * Add post state on edit.php to the page or post that has been altered
	 *
	 * Called outside autodescription_run
	 *
	 * Applies `hmpl_ad_states` filters.
	 *
	 * @uses $this->add_post_state
	 *
	 * @since 2.1.0
	 */
	public function post_state() {

		//* Marked for removal @since 2.3.5
		$allow_states = apply_filters( 'hmpl_ad_states', true );
		//* @since 2.3.0 filter.
		$allow_states = (bool) apply_filters( 'the_seo_framework_allow_states', $allow_states );

		//* Prevent this function from running if this plugin is set to disabled.
		if ( ! $allow_states )
			return;

		add_filter( 'display_post_states', array( $this, 'add_post_state' ) );

	}

	/**
	 * Adds post states in post/page edit.php query
	 *
	 * @param array states 		the current post state
	 * @param string redirected	$this->get_custom_field( 'redirect' );
	 * @param string noindex	$this->get_custom_field( '_genesis_noindex' );
	 *
	 * @since 2.1.0
	 *
	 * @todo convert this to own table column
	 * @todo add search exclude
	 */
	public function add_post_state( $states = array() ) {
		global $post;

		if ( !empty( $post ) ) {
			$post_id = $post->ID;

			$searchexclude = $this->get_custom_field( 'exclude_local_search', $post_id ) ? true : false;

			if ( $searchexclude === true )
				$states[] = __( 'No Search', 'autodescription' );
		}

		return $states;
	}

	/**
	 * Removes the Genesis SEO meta box
	 *
	 * @since 2.0.0
	 *
	 * Removed.
	 * @since 2.2.4
	 */
	public function admin_run() {
		//* Replace Genesis meta boxes with AutoDescription
		$genesis = $this->is_theme( 'genesis' );

		if ( $genesis ) {
			/**
			 * @since 2.0.0
			 */
			remove_action( 'admin_menu', 'genesis_add_inpost_seo_box', 10);

			/**
			 * @since 2.1.8
			 */
			remove_action( 'admin_init', 'genesis_add_taxonomy_seo_options', 10 );
		}

	}

	/**
	 * Removes the Genesis SEO meta boxes on the SEO Settings page
	 *
	 * @since 2.2.4
	 */
	public function no_more_genesis_seo() {
		$plugins = array(
				// Classes to detect.
				'classes' => array(
					'AutoDescription_Load',
				),

				// Functions to detect.
				'functions' => array(),

				// Constants to detect.
				'constants' => array(),
			);

		return (array) $plugins;
	}

	/**
	 * Adds post type support
	 *
	 * @since 2.1.6
	 */
	public function post_type_support() {

		// We're not supporting media library items yet.
		// @todo 'attachment'
		$post_types = array('post', 'page');

		foreach ( $post_types as $type ) {
			add_post_type_support( $type, array( 'autodescription-meta', ) );
		}

	}

	/**
	 * Helper function for allowed post/page screens where this plugin is active.
	 *
	 * @param array $screens The allowed screens
	 *
	 * @since 2.1.9
	 *
	 * @return array $screens
	 */
	protected function supported_screens() {
		$screens = array(
			'edit-post',
			'post',
			'edit-page',
			'page',
			'edit-category',
			'edit-post_tag',
		);

		return $screens;
	}

	/**
	 * Enqueues JS in the admin footer
	 *
	 * @since 2.1.9
	 *
	 * @param $hook the current page
	 */
	public function enqueue_javascript( $hook ) {
		$screen = get_current_screen();
		$id = $screen->id;

		$supported = $this->supported_screens();

		if ( in_array( $id, $supported ) )
			add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_javascript' ), 11 );
	}

	/**
	 * AutoDescription Javascript helper file
	 *
	 * @since 2.0.2
	 *
	 * @usedby add_inpost_seo_box
	 * @usedby enqueue_javascript
	 *
	 * @param string|array|object $hook the current page
	 * @param array|object $term the current term
	 *
	 * @todo Optimize this
	 */
	public function enqueue_admin_javascript( $hook ) {

		$suffix = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';

		wp_enqueue_script( 'autodescription-js', THE_SEO_FRAMEWORK_DIR_URL . "lib/js/autodescription{$suffix}.js", array( 'jquery' ), THE_SEO_FRAMEWORK_VERSION, true );

		$blog_name = get_bloginfo( 'name', 'display' );

		/**
		 * We're gaining UX in exchange for resource usage.
		 *
		 * Any way to cache this?
		 *
		 * @since 2.2.4
		 */
		if ( $hook ) {
			// We're somewhere within default WordPress pages.
			$post_id = get_the_ID();

			if ( $post_id ) {
				// We're on post.php
				$title = $this->title( '', '', '', '', '', '', true, true );
				$additions = $blog_name;
			} else {
				// We're in terms and taxonomies
				// Can't fetch title.
				$title = '';
				$additions = $blog_name;
			}

		} else {
			// We're on our SEO settings pages.
			if ( 'page' === get_option( 'show_on_front' ) ) {
				// Home is a page.
				$inpost_title = $this->get_custom_field( '_genesis_title', get_option( 'page_on_front' ) );
			} else {
				// Home is a blog.
				$input_title = '';
			}
			$title = !empty( $inpost_title ) ? $inpost_title : $blog_name;
			$additions = get_bloginfo( 'description', 'raw' );
		}

		$strings = array(
			'saveAlert'		=> __( 'The changes you made will be lost if you navigate away from this page.', 'autodescription' ),
			'confirmReset'	=> __( 'Are you sure you want to reset all SEO settings to their defaults?', 'autodescription' ),
			'siteTitle' 	=> $title,
			'titleAdditions' => $additions,
			'titleSep' 		=> $this->get_option( 'title_seperator' ),
		);

		wp_localize_script( 'autodescription-js', 'autodescriptionL10n', $strings );
    }

	/**
	 * Enqueues CSS in the admin header
	 *
	 * @since 2.1.9
	 *
	 * @param $hook the current page
	 */
	public function enqueue_css( $hook ) {
		$screen = get_current_screen();
		$id = $screen->id;

		$supported = $this->supported_screens();

		if ( in_array( $id, $supported ) )
			add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_css' ), 11 );
	}

	/**
	 * CSS for the AutoDescription Bar
	 *
	 * @since 2.1.9
	 *
	 * @param $hook the current page
	 *
	 * @todo get_network_option
	 */
	public function enqueue_admin_css( $hook ) {

		$rtl = '';

		if ( is_rtl() )
			$rtl = '-rtl';

		$suffix = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';

		wp_enqueue_style( 'autodescription-css', THE_SEO_FRAMEWORK_DIR_URL . "lib/css/autodescription{$rtl}{$suffix}.css", array(), THE_SEO_FRAMEWORK_VERSION, 'all' );
	}

	/**
	 * Mark up content with code tags.
	 *
	 * Escapes all HTML, so `<` gets changed to `&lt;` and displays correctly.
	 *
	 * @since 2.0.0
	 *
	 * @param  string $content Content to be wrapped in code tags.
	 *
	 * @return string Content wrapped in code tags.
	 */
	public function code_wrap( $content ) {
		return '<code>' . esc_html( $content ) . '</code>';
	}

	/**
	 * Mark up content with code tags.
	 *
	 * Escapes no HTML.
	 *
	 * @since 2.2.2
	 *
	 * @param  string $content Content to be wrapped in code tags.
	 *
	 * @return string Content wrapped in code tags.
	 */
	public function code_wrap_noesc( $content ) {
		return '<code>' . $content . '</code>';
	}

	/**
	 * Return custom field post meta data.
	 *
	 * Return only the first value of custom field. Return false if field is
	 * blank or not set.
	 *
	 * @since 2.0.0
	 *
	 * @param string $field	Custom field key.
	 * @param int $post_id	The post ID
	 *
	 * @return string|boolean Return value or false on failure.
	 *
	 * @thanks StudioPress :)
	 *
	 * @uses cache
	 * @since 2.2.5
	 */
	public function get_custom_field( $field, $post_id = null ) {

		//* No field has been provided.
		if ( empty( $field ) )
			return false;

		//* Setup cache.
		static $field_cache = array();

		//* Check field cache.
		if ( isset( $field_cache[$field][$post_id] ) )
			//* Field has been cached.
			return $field_cache[$field][$post_id];

		if ( null === $post_id )
			$post_id = get_the_ID();

		if ( null === $post_id )
			return '';

		$custom_field = get_post_meta( $post_id, $field, true );

		// If custom field is empty, return null.
		if ( ! $custom_field )
			$field_cache[$field][$post_id] = '';

		//* Render custom field, slashes stripped, sanitized if string
		$field_cache[$field][$post_id] = is_array( $custom_field ) ? stripslashes_deep( $custom_field ) : stripslashes( wp_kses_decode_entities( $custom_field ) );

		return $field_cache[$field][$post_id];
	}

	/**
	 * Checks the screen hook.
	 *
	 * @since 2.2.2
	 *
	 * @return bool true if screen match.
	 */
	public function is_menu_page( $pagehook = '' ) {
		global $page_hook;

		if ( isset( $page_hook ) && $page_hook === $pagehook )
			return true;

			//* May be too early for $page_hook
		if ( isset( $_REQUEST['page'] ) && $_REQUEST['page'] === $pagehook )
			return true;

		return false;
	}

	/**
	 * Redirect the user to an admin page, and add query args to the URL string
	 * for alerts, etc.
	 *
	 * @since 2.2.2
	 *
	 * @param string $page			Menu slug.
	 * @param array  $query_args 	Optional. Associative array of query string arguments
	 * 								(key => value). Default is an empty array.
	 *
	 * @return null Return early if first argument is false.
	 */
	public function admin_redirect( $page, array $query_args = array() ) {

		if ( ! $page )
			return;

		$url = html_entity_decode( menu_page_url( $page, 0 ) );

		foreach ( (array) $query_args as $key => $value ) {
			if ( empty( $key ) && empty( $value ) ) {
				unset( $query_args[$key] );
			}
		}

		$url = add_query_arg( $query_args, $url );

		wp_redirect( esc_url_raw( $url ) );

	}

	/**
	 * Google docs language determinator.
	 *
	 * @since 2.2.2
	 *
	 * @return string language code
	 */
	protected function google_language() {

		/**
		 * Cache value
		 * @since 2.2.4
		 */
		static $language = null;

		if ( isset( $language ) )
			return $language;

		//* Language shorttag to be used in Google help pages,
		$language = _x( 'en', 'e.g. en for English, nl for Dutch, fi for Finish, de for German', 'autodescription' );

		return $language;
	}

}
