<?php
/**
 * Plugin Name: Screets Live Chat (shared on wplocker.com)
 * Plugin URI: http://www.screets.com
 * Description: <strong>Starter.</strong> Chat with your visitors in your website.
 * Version: 2.0.1
 * Author: Screets Team
 * Author URI: http://www.screets.com
 * Requires at least: 4.0
 * Tested up to: 4.4
 *
 *
 * Text Domain: schat
 * Domain Path: /languages/
 *
 * @package Screets Live Chat
 * @category Core
 * @author Screets
 *
 * COPYRIGHT (c) 2016 Screets. All rights reserved.
 * This  is  commercial  software,  only  users  who have purchased a valid
 * license  and  accept  to the terms of the  License Agreement can install
 * and use this program.
 */

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

global $wpdb;

// Define Constants
define( 'SCHAT_PATH', untrailingslashit( plugin_dir_path( __FILE__ ) ) );
define( 'SCHAT_URL', plugins_url( basename( plugin_dir_path( __FILE__ ) ), basename( __FILE__ ) ) );
define( 'SCHAT_NOW', current_time( 'timestamp' ) );
define( 'SCHAT_SLUG', 'screets-lc' );
define( 'SCHAT_PX', $wpdb->prefix . 'schat_' ); // DB Table prefix

define( 'SCHAT_FIREBASE_VERSION', '2.3.2' );

// Add "SCHAT_PHP_SESSIONS" constant in your wp-config.php file to change this value
if( !defined( 'SCHAT_PHP_SESSIONS' ) )
	define( 'SCHAT_PHP_SESSIONS', true );

if ( ! class_exists( 'SCHAT' ) ) {

class SCHAT {

	/**
	 * @var object
	 */
	var $option;

	/**
	 * @var array
	 */
	var $opts = array();

	/**
	 * @var object
	 */
	var $session;

	/**
	 * @var array
	 */
	var $user = array();

	/**
	 * @var array
	 */
	var $meta = array();

	/**
	 * @var string
	 */
	var $min_jquery = '1.8';

	/**
	 * @var string
	 */
	var $app_token;


	/**
	 * @var array
	 */
	var $admin_notices = array();

	/**
	 * Chat Constructor
	 *
	 * @access public
	 * @return void
	 */
	public function __construct() {

		// Check that function get_plugin_data exists
		if ( !function_exists( 'get_plugin_data' ) )
			require_once( ABSPATH . 'wp-admin/includes/plugin.php' );

		// Get plugin meta
		$this->meta = get_plugin_data( __FILE__, false );

		// Define app meta constants
		define( 'SCHAT_NAME', $this->meta['Name'] );
		define( 'SCHAT_NAME_SHORT', str_replace( 'Screets ', '', SCHAT_NAME ) );
		define( 'SCHAT_VERSION', $this->meta['Version'] );

		// Install plugin
		register_activation_hook( __FILE__, array( $this, 'activate' ) );

		// Updates
		add_action( 'admin_init', array( &$this, 'admin_init' ), 5 );

		// Include required files
		$this->includes();

		// Setup session
		$this->session = new SCHAT_Session();

		// Actions
		add_action( 'init', array( &$this, 'init' ), 0 );
		add_action( 'plugins_loaded', array( &$this, 'load_plugin_textdomain' ) );

		// Loaded action
		do_action( 'schat_loaded' );

	}


	/**
	 * Init Screets Screets Live Chat when WordPress Initialises
	 *
	 * @access public
	 * @return void
	 */
	function init() {

		// Before init action
		do_action( 'before_schat_init' );

		// Check if Woocommerce installed
		if( function_exists( 'is_woocommerce' ) ) {
			define( 'SCHAT_WC_INSTALLED', true );
		}

		// Some user info
		$this->current_page = schat_current_page_url();
		$this->ip = schat_ip_address();

		// User is operator?
		if( current_user_can( 'answer_visitors' ) ) {
			define( 'SCHAT_OP', true );

		// User is visitor?
		} else
			define( 'SCHAT_VISITOR', true );


		if ( ! is_admin() || defined( 'DOING_AJAX' ) ) {

			add_action( 'wp_enqueue_scripts', array(&$this, 'frontend_scripts') );
			add_action( 'wp_print_scripts', array( $this, 'check_jquery' ), 25 );

		}

		// Create options object
		$this->option = new SCHAT_options( __FILE__ );

		// Get options set
		require SCHAT_PATH . '/core/options/options.php';

		// Create options page
		$this->option->add_options_page(
			array(
				'menu_title' => __( 'Options', 'schat' ),
				'parent' => 'screets_lc'
			),
			$schat_opts_set
		);

		// Get all options
		$this->opts = $this->option->get_options();

		// Ajax requests
		$ajax_prefix = ( !empty( $this->opts['faster_ajax'] ) ) ? 'schat_' : 'wp_';
		add_action( $ajax_prefix . 'ajax_schat_ajax_callback', 'schat_ajax_callback' );
		add_action( $ajax_prefix . 'ajax_nopriv_schat_ajax_callback', 'schat_ajax_callback' );

		// WPML support (updates opts)
		if( function_exists( 'icl_register_string' ) )
			$this->WPML( $schat_opts_set );

		// Save user info
		if( is_user_logged_in() ) {
			global $current_user;

			// Get currently logged user info
			get_currentuserinfo();

			$this->user = $current_user;

		// Visitor info
		} else {

			$visitor_id = $this->session->get( 'visitor_id' );

			// Create new unique ID
			if( empty( $visitor_id ) ) {
				$visitor_id = uniqid( rand(), false );

				// Save id into the session
				$this->session->set( 'visitor_id', $visitor_id );

			}

			$this->user = (object) array(
				'ID' => $visitor_id,
				'display_name' => null,
				'user_email' => null
			);

		}

		// Register Visitors post type when default AJAX base is enabled
		/*if( $this->opts['app'] == 'none' ) {

			$labels = array(
				'name'                => _x( 'Visitors', 'Post Type General Name', 'schat' ),
				'singular_name'       => _x( 'Visitor', 'Post Type Singular Name', 'schat' ),
				'menu_name'           => __( 'Visitor', 'schat' ),
				'parent_item_colon'   => __( 'Parent Visitor:', 'schat' ),
				'all_items'           => __( 'All Visitors', 'schat' ),
				'view_item'           => __( 'View Visitor', 'schat' ),
				'add_new_item'        => __( 'Add New Visitor', 'schat' ),
				'add_new'             => __( 'New Visitor', 'schat' ),
				'edit_item'           => __( 'Edit Visitor', 'schat' ),
				'update_item'         => __( 'Update Visitor', 'schat' ),
				'search_items'        => __( 'Search visitors', 'schat' ),
				'not_found'           => __( 'No visitors found', 'schat' ),
				'not_found_in_trash'  => __( 'No visitors found in Trash', 'schat' ),
			);
			$args = array(
				'labels'              => $labels,
				'supports'            => array( 'title', 'custom-fields' ),
				'hierarchical'        => false,
				'public'              => false,
				'show_ui'             => true,
				'show_in_menu'        => false,
				'show_in_nav_menus'   => false,
				'show_in_admin_bar'   => false,
				'menu_position'       => 60,
				'menu_icon'           => '',
				'can_export'          => true,
				'has_archive'         => true,
				'exclude_from_search' => true,
				'publicly_queryable'  => true,
				'capability_type'     => 'page',
				'capabilities' 		  => array(
					// 'create_posts' => false
				)
			);
			register_post_type( 'schat_visitor', $args );

		}*/

		// Register Offline Messages post type
		$labels = array(
			'name'                => _x( 'Offline Messages', 'Post Type General Name', 'schat' ),
			'singular_name'       => _x( 'Offline Message', 'Post Type Singular Name', 'schat' ),
			'menu_name'           => __( 'Offline Message', 'schat' ),
			'parent_item_colon'   => __( 'Parent Offline Message:', 'schat' ),
			'all_items'           => __( 'All Offline Messages', 'schat' ),
			'view_item'           => __( 'View Offline Message', 'schat' ),
			'add_new_item'        => __( 'Add New Offline Message', 'schat' ),
			'add_new'             => __( 'New Offline Message', 'schat' ),
			'edit_item'           => __( 'Edit Offline Message', 'schat' ),
			'update_item'         => __( 'Update Offline Message', 'schat' ),
			'search_items'        => __( 'Search offline message', 'schat' ),
			'not_found'           => __( 'No offline message found', 'schat' ),
			'not_found_in_trash'  => __( 'No offline message found in Trash', 'schat' ),
		);

		$args = array(
			'labels'              => $labels,
			'supports'            => array( 'title' ),
			'hierarchical'        => false,
			'public'              => false,
			'show_ui'             => true,
			'show_in_menu'        => false,
			'show_in_nav_menus'   => false,
			'show_in_admin_bar'   => false,
			'menu_position'       => 60,
			'menu_icon'           => '',
			'can_export'          => true,
			'has_archive'         => false,
			'exclude_from_search' => true,
			'publicly_queryable'  => false,
			'rewrite' 			  => false,
			'capability_type'     => 'page',
			'capabilities' 		  => array(
				// 'create_posts' => false
			)
		);
		register_post_type( 'schat_offline_msg', $args );

		// Predefined Messages post type
		/*$labels = array(
			'name'                => _x( 'Predifined Messages', 'Post Type General Name', 'schat' ),
			'singular_name'       => _x( 'Predefined Message', 'Post Type Singular Name', 'schat' ),
			'menu_name'           => __( 'Predefined Message', 'schat' ),
			'parent_item_colon'   => __( 'Parent Predefined Message:', 'schat' ),
			'all_items'           => __( 'All Predefined Messages', 'schat' ),
			'view_item'           => __( 'View Predefined Message', 'schat' ),
			'add_new_item'        => __( 'Add New Predefined Message', 'schat' ),
			'add_new'             => __( 'New Predefined Message', 'schat' ),
			'edit_item'           => __( 'Edit Predefined Message', 'schat' ),
			'update_item'         => __( 'Update Predefined Message', 'schat' ),
			'search_items'        => __( 'Search predefined message', 'schat' ),
			'not_found'           => __( 'No predefined message found', 'schat' ),
			'not_found_in_trash'  => __( 'No predefined message found in Trash', 'schat' ),
		);
		$args = array(
			'labels'              => $labels,
			'supports'            => array( 'title', 'editor' ),
			'hierarchical'        => false,
			'public'              => false,
			'show_ui'             => true,
			'show_in_menu'        => false,
			'show_in_nav_menus'   => false,
			'show_in_admin_bar'   => false,
			'menu_position'       => 60,
			'menu_icon'           => '',
			'can_export'          => true,
			'has_archive'         => true,
			'exclude_from_search' => true,
			'publicly_queryable'  => true,
			'capability_type'     => 'page',
			'capabilities' 		  => array(
				// 'create_posts' => false
			)
		);
		register_post_type( 'schat_predefined_msg', $args );*/

		// Add operator name to user fields
		if( defined( 'SCHAT_OP' ) ) {
			add_action( 'show_user_profile', array(&$this, 'xtra_profile_fields'), 10 );
			add_action( 'edit_user_profile', array(&$this, 'xtra_profile_fields'), 10 );
			add_action( 'personal_options_update', array(&$this, 'save_xtra_profile_fields') );
			add_action( 'edit_user_profile_update', array(&$this, 'save_xtra_profile_fields') );
		}

		// Add custom css
		if( !empty( $this->opts['custom_css'] ) || @$this->opts['avatar_size'] != 30 )
			add_action( 'wp_footer', array( &$this, 'custom_css' ) );

		// Render chat box
		add_action( 'wp_footer', array( &$this, 'display_chatbox') );

		// Shortcodes
		add_shortcode( 'schat_btn', 'schat_shortcode_open_chatbox' );

		// Check updates
		$this->update();

		// Initialization action
		do_action( 'schat_init' );

	}

	/**
	 * Initialization Screets Screets Live Chat for back-end
	 *
	 * @access public
	 * @return void
	 */
	function admin_init() {

		// Load back-end styles and scripts
		add_action( 'admin_enqueue_scripts', array( &$this, 'backend_scripts' ) );

		// Get current page
		$current_page = ( !empty( $_GET['page'] ) ) ? $_GET['page'] : null;

		// Check CX setup
		if( $current_page  == SCHAT_SLUG or $current_page  == 'screets_lc' ) {

			require SCHAT_PATH . '/core/fn.setup.php';

			// Check CX configuration
			schat_check_setup();


		}

		// Settings page
		if( $current_page == SCHAT_SLUG ) {

			$l = ( !empty( $this->opts['license_key'] ) ) ? $this->opts['license_key'] : null;

		}

	}

	/**
	 * Include required core files
	 *
	 * @access public
	 * @return void
	 */
	function includes() {

		// Back-end includes
		if(  is_admin())  $this->admin_includes();

		// Include core files
		require SCHAT_PATH . '/core/lib/firebaseToken.php';
		require SCHAT_PATH . '/core/class.options.php';
		require SCHAT_PATH . '/core/class.user.php';
		require SCHAT_PATH . '/core/class.session.php';
		require SCHAT_PATH . '/core/fn.firebase.php';
		require SCHAT_PATH . '/core/fn.visitor.php';
		require SCHAT_PATH . '/core/fn.common.php';
		require SCHAT_PATH . '/core/fn.ajax.php';
		require SCHAT_PATH . '/core/fn.security.php';
		require SCHAT_PATH . '/core/fn.formatting.php';
		require SCHAT_PATH . '/core/fn.offline.php';
		require SCHAT_PATH . '/core/fn.shortcodes.php';

	}

	/**
	 * Include required admin files
	 *
	 * @access public
	 * @return void
	 */
	function admin_includes() {

		// Include admin files
		require SCHAT_PATH . '/core/fn.admin.php';
		require SCHAT_PATH . '/core/class.logs.php';

	}

	/**
	 * Get Ajax URL
	 *
	 * @access public
	 * @return string
	 */
	function ajax_url() {

		if( !empty( $this->opts['faster_ajax'] ) )
			return str_replace( array('https:', 'http:'), '', SCHAT_URL . '/core/schat.ajax.php' );
		else
			return str_replace( array('https:', 'http:'), '', admin_url( 'admin-ajax.php' ) );

	}

	/**
	 * Localization
	 *
	 * @access public
	 * @return void
	 */
	function load_plugin_textdomain() {

		load_plugin_textdomain( 'schat', false, SCHAT_PATH . '/languages/' );

	}

	/**
	 * Add xtra profile fields
	 *
	 * @access public
	 * @return void
	 */
	function xtra_profile_fields( $user ) { ?>

		<h3><?php _e( 'Chat Options', 'schat' ); ?></h3>

		<table class="form-table">

			<tr>
				<th><?php _e( 'Operator Name', 'schat' ); ?></th>
				<td>
					<input type="text" name="schat_op_name" id="f_chat_op_name" value="<?php echo esc_attr( get_the_author_meta( 'schat_op_name', $user->ID ) ); ?>" class="regular-text" />
					<br>
					<span class="description"><?php _e( 'Refresh chat console page when you update operator name', 'schat' ); ?></span>
				</td>
			</tr>


		</table>


	<?php }


	/**
	 * Save xtra profile fields
	 *
	 * @access public
	 * @return void
	 */
	function save_xtra_profile_fields( $user_id ) {

		if ( !current_user_can( 'edit_user', $user_id ) )
			return false;

		// Op name isn't defined yet, create new one for user
		if( empty( $_POST['schat_op_name'] ) ) {

			$op_name = $this->user->display_name;


		// OP name
		} else
			$op_name = $_POST['schat_op_name'];


		// Update user meta now
		update_user_meta( $user_id, 'schat_op_name', $op_name );

	}


	/**
	 * Activate plugin
	 *
	 * @access public
	 * @return void
	 */
	public function activate() {
		global $wpdb;

		require SCHAT_PATH . '/core/fn.upgrade.php';

		// Get license key
		$license = !empty( $this->opts['license_key'] ) ? $this->opts['license_key'] : null;

		// Upgrade plugin
		schat_upgrade( $license );

		// Update operator role
		schat_update_op_role( 'editor' );

		/**
		 * Administration role
		 */
		$admin_role = get_role( 'administrator' );
		$admin_role->add_cap( 'answer_visitors' );

		/**
		 * Chat Operator role
		 */
		$op_role = get_role( 'schat_op' );
		$op_role->add_cap( 'answer_visitors' );

	}

	/**
	 * The plugin requires jQuery 1.8
	 * If, by the time wp_print_scrips is called, jQuery is outdated (i.e not
	 * using the version in core) we need to deregister it and register the
	 * core version of the file.
	 *
	 * @access public
	 * @return void
	 */
	public function check_jquery() {
		global $wp_scripts;

		// Enforce minimum version of jQuery
		if ( ! empty( $wp_scripts->registered['jquery']->ver ) && ! empty( $wp_scripts->registered['jquery']->src ) && $wp_scripts->registered['jquery']->ver < '1.7' ) {
			wp_deregister_script( 'jquery' );
			wp_register_script( 'jquery', '/wp-includes/js/jquery/jquery.js', array(), $this->min_jquery );
			wp_enqueue_script( 'jquery' );
		}
	}

	/**
	 * Display chatbox
	 *
	 * @access public
	 * @return void
	 */
	function display_chatbox() {

		// Get plugin options
		$plugin_opts = schat_get_jquery_plug_opts();
		?>

		<div id="SCHAT_chatbox"></div>

		<script type="text/javascript">
		(function ($) {

			$(document).ready(function () {

				var el = $('#SCHAT_chatbox');

				/**
				 * Initialize Screets Chat plugin
				 */
				el.schat({
					<?php schat_print_custom_opts( $plugin_opts ); ?>
				});

			});

		} (window.jQuery || window.Zepto));

		</script>

	<?php
	}

	/**
	 * Add WPML support to the plguin
	 *
	 * @return void
	 */
	function WPML( $opts_set ) {

		foreach( $opts_set as $opt ) {


			if( !empty( $opt['translate'] ) ) {

				// Register strings to WPML
				icl_register_string( 'Screets CX', $opt['name'], $this->opts[ $opt['id'] ] );

				// Update translations in options
				$this->opts[ $opt['id'] ] = icl_t( 'Screets CX', $opt['name'], $this->opts[ $opt['id'] ] );

			}
		}
	}


	/**
	 * Add Custom CSS
	 *
	 * @access public
	 * @return void
	 */

	function custom_css() { ?>

	    <style type="text/css">
	    	/* <?php echo SCHAT_NAME; ?> custom CSS */
	    	<?php
	    		echo $this->opts['custom_css'];

	    		// Avatar size
	    		$avatar_size = @$this->opts['avatar_size'] || 30;
	    		$avatar_radius = @$this->opts['avatar_radius'] || 30;

	    		// Update Avatar Size?
	    		if( $avatar_size != 30 ) {
	    			echo '.schat-cnv .schat-avatar, .schat-cnv .schat-avatar.schat-img img { width: ' . $avatar_size . 'px; }'
	    				.'.schat-cnv-msg { margin-left: ' . ( $avatar_size + 10 ) . 'px; }';
	    		}

				// Update avatar radius
				if( $avatar_radius != 30 ) {
					echo '.schat-cnv .schat-avatar.schat-img img { border-radius: ' . $avatar_radius . ' px; }';
				}
	    	?>
    	</style>

	<?php
	}


	/**
	 * Front-end styles and scripts
	 *
	 * @access public
	 * @return void
	 */
	function frontend_scripts() {

		$suffix_css = ( !empty( $this->opts['compress-group']['compress_css'] ) ) ? '.min' : '';

		// Base template stylesheet
		if( empty( $this->opts['compress-group']['disable_css'] ) ) {
			wp_register_style(
				'schat-basic',
				plugins_url( 'assets/css', __FILE__ )  . '/schat.basic' . $suffix_css . '.css'
			);
			wp_enqueue_style( 'schat-basic' );
		}

		// Use jQuery
		wp_enqueue_script( 'jquery' );

		// Application JS
		$this->load_app_js();

		// Load common admin scripts if user is admin / operator
		if( defined( 'SCHAT_OP' ) )
			$this->common_admin_scripts();

		// § Plug-in
		wp_register_script(
			'jquery-autosize',
			SCHAT_URL . '/assets/js/lib/jquery.autosize.min.js',
			array( 'jquery' ),
			'1.17.1'
		);
		wp_enqueue_script( 'jquery-autosize' );

	}


	/**
	 * Back-end styles and scripts
	 *
	 * @access public
	 * @return void
	 */
	function backend_scripts() {

		$page = '';
		$suffix_css = ( !empty( $this->opts['compress-group']['compress_css'] ) ) ? '.min' : '';

		// Get currently logged user info
		get_currentuserinfo();

		// Application JS
		$this->load_app_js();

		// Get current page
		if( !empty( $_REQUEST['page'] ) )
			$page = $_REQUEST['page'];

		// Load in chat console
		if( $page == 'screets_lc'  ) {

			// Console stylesheet
			wp_register_style(
				'schat-console',
				SCHAT_URL . '/assets/css/schat.console.css'
			);
			wp_enqueue_style( 'schat-console' );

			// AutoSize Plug-in
			wp_register_script(
				'jquery-autosize',
				SCHAT_URL . '/assets/js/lib/jquery.autosize.min.js',
				array( 'jquery' ),
				'1.17.1'
			);
			wp_enqueue_script( 'jquery-autosize' );

			// Tipsy Plug-in
			wp_register_script(
				'jquery-tipsy',
				SCHAT_URL . '/assets/js/lib/jquery.tipsy.min.js',
				array( 'jquery' ),
				'1.0'
			);
			wp_enqueue_script( 'jquery-tipsy' );

			// Console JS
			wp_register_script(
				'schat-console',
				SCHAT_URL . '/assets/js/schat.console.js',
				array( 'jquery' ),
				SCHAT_VERSION
			);
			wp_enqueue_script( 'schat-console' );

		}

		// Load common admin scripts if user is admin / operator
		if( defined( 'SCHAT_OP' ) )
			$this->common_admin_scripts();

	}

	/**
	 * Common styles and scripts for admin / operators
	 *
	 * @access public
	 * @return void
	 */
	function common_admin_scripts() {}

	/**
	 * Update CX and check for new updates from Screets Server
	 *
	 * @access public
	 * @return void
	 */
	function update() {

		require SCHAT_PATH . '/core/lib/update-checker/plugin-update-checker.php';

		$checker = PucFactory::buildUpdateChecker(
			'http://screets.org/apps/api/v1/wp-updates/?action=get_metadata&slug=' . SCHAT_SLUG,
			__FILE__,
			SCHAT_SLUG
		);

		$checker->addQueryArgFilter( array( &$this, '_filter_updates' ) );

	}

	/**
	 * Filter updates with API key
	 *
	 * @access public
	 * @return array Query arguments
	 */
	function _filter_updates( $query_args ) {

		$api_key = $this->opts['license_key'];

		// Include API key
		if ( !empty( $api_key ) ) {
			$query_args['key'] = $api_key;
			$query_args['domain'] = schat_current_domain();
		}

		return $query_args;

	}

	/**
	 * Authentication user
	 *
	 * @access public
	 * @return string Auth token
	 */
	public function auth() {
		global $wpdb;

		if( empty( $this->opts[ 'app_token' ] ) )
			return;

		// FireBase authentication
		$token_gen = new Services_FirebaseTokenGenerator( $this->opts[ 'app_token' ] );

		// Administrator user?
		$is_admin = ( current_user_can( 'manage_options' ) ) ? true : false;

		// Is Operator?
		$is_op = ( defined( 'SCHAT_OP' ) ) ? true : false;

		$prefix = ( is_user_logged_in() && !defined( 'SCHAT_OP' ) ) ? 'usr-' : '';

		// An object or array of data you wish
        // to associate with the token. It will
  		// be available as the variable "auth" in
  		// the Firebase rules engine.
		$data = array(
			'user_id' 		=> $prefix . $this->user->ID,
			'is_operator'	=> $is_op
		);

		$debug = ( !empty( $this->opts['debug'] ) ) ? true : false;

		// Options
		$opts = array(
			// Set to true if you want this
			// token to bypass all security rules.
			'admin'	=> $is_admin,

			// Set to true if you want to enable debug
			// output from your security rules.
			'debug'	=> $debug


			//'expires'	=> 0,			// Set to a number (seconds
                                    	// since epoch) or a DateTime object that
                                    	// specifies the time at which the token
                                    	// should expire.

			//'notBefore'	=> null 	// Set to a number (seconds
                                    	// since epoch) or a DateTime object that
                                    	// specifies the time before which the
                                    	// should be rejected by the server.
		);

		// Create secure auth token
		return $token_gen->createToken( $data, $opts );

	}


	/**
	 * Load application JS
	 *
	 * @access public
	 * @return void
	 */
	function load_app_js() {

		$suffix_js = ( !empty( $this->opts['compress-group']['compress_js'] ) ) ? '.min' : '';

		// Firebase CDN
		wp_register_script(
			'firebase',
			SCHAT_URL . '/assets/js/firebase.js',
			null,
			SCHAT_FIREBASE_VERSION
		);
		wp_enqueue_script( 'firebase' );

		/**
		 * Application JS
		 */
		wp_register_script(
			'schat-app',
			SCHAT_URL . '/assets/js/schat.app' . $suffix_js . '.js',
			array( 'jquery', 'firebase' ),
			SCHAT_VERSION
		);
		wp_enqueue_script( 'schat-app' );

		$company_avatar = !empty( $this->opts['default_avatar'] ) ? $this->opts['default_avatar'] : '';

		// Custom Data
		$js_vars = array(
			'ajax_url'   		=> $this->ajax_url(),
			'plugin_url'   		=> SCHAT_URL,
			'is_front_end' 		=> ( !is_admin() ) ? true : null,
			'is_op' 			=> ( defined( 'SCHAT_OP' ) && is_admin() ) ? true : null,
			'is_home' 			=> ( is_home() || is_front_page() ) ? true : null,
			'current_page'		=> $this->current_page,
			'company_avatar'	=> $company_avatar,
			'ip' 				=> $this->ip
		);

		// Add user information
		if( is_user_logged_in() ) {

			// Get user prefix
			$user_prefix = ( defined( 'SCHAT_OP' ) && is_admin() ) ? 'op-' : '';

			$js_vars['user_id'] = $user_prefix . $this->user->ID;
			$js_vars['user_name'] = schat_get_operator_name();
			$js_vars['user_email'] = $this->user->user_email;
			$js_vars['user_email_hash'] = md5( $this->user->user_email );

		}

		if( defined( 'SCHAT_OP' ) ) {

			// Get Firebase application ID and token
			$js_vars['app_id'] = !empty( $this->opts['app_url'] ) ? $this->opts['app_url'] : null;

			// Get current page
			$current_page = ( !empty( $_GET['page'] ) ) ? $_GET['page'] : null;

			// Console messages
			if( $current_page == 'screets_lc' ) {

				$js_vars['msgs'] = array(
					'connect' => __( 'Connect', 'schat' ),
					'disconnect' => __( 'Disconnect', 'schat' ),
					'you_offline' => __( "You're offline", 'schat' ),
					'save_note' => __( 'Chat logs saved into your own server and removed from realtime app platform after ending chat', 'schat' ),
					'save_end_chat' => __( 'Save and end chat', 'schat' ),
					'ntf_close_console' => __( 'If you leave, you will be logged out of chat. However you will be able to save conversations into your server when you come back to console!', 'schat' ),
					'new_msg' => __( 'New Message', 'schat' ),
					'new_user_online' => __( 'New User Online', 'schat' ),
					'saving' => __( 'Saving', 'schat' )
				);

			}
		}

		wp_localize_script( 'schat-app', 'schat', $js_vars );
	}
}

// Init Chat class
$GLOBALS['SCHAT'] = new SCHAT();

} // class_exists


// cok sevmistik, keske erimeseydik...
